<?php

namespace Tests;

use App\Models\User;
use App\Models\Client;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Inventory;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OrderFlowTest extends TestCase
{
    use RefreshDatabase;

    protected $user;
    protected $client;
    protected $boxes = ['BOX001', 'BOX002'];

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test user
        $this->user = User::factory()->create();
        $this->actingAs($this->user);
        
        // Create test client
        $this->client = Client::factory()->create([
            'client_name' => 'Cliente Prueba',
            'email' => 'test@example.com',
        ]);
        
        // Create test inventory items
        foreach ($this->boxes as $boxNumber) {
            for ($i = 0; $i < 3; $i++) {
                Inventory::factory()->create([
                    'n_carton' => $boxNumber,
                    'full_barcode' => 'TEST' . $boxNumber . $i,
                    'mocaco' => 'TEST' . $boxNumber,
                    'status' => 'disponible',
                    'n_id_operario' => 'TEST_OP',
                ]);
            }
        }
    }

    /** @test */
    public function test_create_draft_order_reserves_inventory()
    {
        $response = $this->postJson('/api/order-preparation/save-draft', [
            'client_id' => $this->client->id,
            'operario_id' => 'OP001',
            'notes' => 'Pedido de prueba',
            'boxes' => $this->boxes,
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $order = Order::latest()->first();
        $this->assertNotNull($order);
        $this->assertEquals('draft', $order->status);
        
        // Verify inventory items are RESERVED
        $reservedItems = Inventory::whereIn('n_carton', $this->boxes)
            ->where('status', 'reservado')
            ->get();
            
        $this->assertCount(6, $reservedItems, 'Debería haber 6 items reservados (3 por cada caja)');
        
        // Verify order_id is set
        foreach ($reservedItems as $item) {
            $this->assertEquals($order->id, $item->order_id);
        }
        
        // Verify OrderItems were created
        $orderItems = OrderItem::where('order_id', $order->id)->get();
        $this->assertCount(6, $orderItems, 'Debería haber 6 items en el pedido');
    }

    /** @test */
    public function test_confirm_dispatch_changes_inventory_to_dispatched()
    {
        // Create draft order
        $this->postJson('/api/order-preparation/save-draft', [
            'client_id' => $this->client->id,
            'operario_id' => 'OP001',
            'boxes' => $this->boxes,
        ]);
        
        $order = Order::latest()->first();
        $orderId = $order->id;
        
        // Get inventory IDs that should be dispatched
        $orderItems = OrderItem::where('order_id', $orderId)->get();
        $inventoryIds = $orderItems->pluck('inventory_id')->toArray();
        
        // Confirm dispatch
        $response = $this->postJson('/api/order-preparation/confirm-dispatch', [
            'order_id' => $orderId,
        ]);
        
        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        // Verify inventory status changed to 'dispatched'
        $dispatchedItems = Inventory::whereIn('id', $inventoryIds)
            ->where('status', 'dispatched')
            ->get();
            
        $this->assertCount(6, $dispatchedItems, 'Todos los items deberían estar como dispatched');
        
        // Verify order status
        $order->refresh();
        $this->assertEquals('dispatched', $order->status);
        $this->assertNotNull($order->dispatched_date);
        
        // Verify inventory was NOT deleted
        $this->assertDatabaseHas('inventory', [
            'id' => $inventoryIds[0],
            'status' => 'dispatched',
        ]);
    }

    /** @test */
    public function test_cancel_order_restores_inventory_to_disponible()
    {
        // Create draft order
        $this->postJson('/api/order-preparation/save-draft', [
            'client_id' => $this->client->id,
            'operario_id' => 'OP001',
            'boxes' => $this->boxes,
        ]);
        
        $order = Order::latest()->first();
        $orderId = $order->id;
        
        $orderItems = OrderItem::where('order_id', $orderId)->get();
        $inventoryIds = $orderItems->pluck('inventory_id')->toArray();
        
        // Cancel order
        $response = $this->postJson('/api/order-preparation/cancel-order', [
            'order_id' => $orderId,
        ]);
        
        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        // Verify inventory restored to 'disponible'
        $availableItems = Inventory::whereIn('id', $inventoryIds)
            ->where('status', 'disponible')
            ->whereNull('order_id')
            ->get();
            
        $this->assertCount(6, $availableItems, 'Todos los items deberían volver a disponible');
        
        // Verify order is cancelled
        $order->refresh();
        $this->assertEquals('cancelled', $order->status);
    }

    /** @test */
    public function test_inventory_shows_reserved_status_in_list()
    {
        // Create draft order to reserve items
        $this->postJson('/api/order-preparation/save-draft', [
            'client_id' => $this->client->id,
            'operario_id' => 'OP001',
            'boxes' => ['BOX001'],
        ]);
        
        $order = Order::latest()->first();
        $orderItems = OrderItem::where('order_id', $order->id)->get();
        $inventoryIds = $orderItems->pluck('inventory_id')->toArray();
        
        // Verify reserved items show in inventory list
        $reserved = Inventory::whereIn('id', $inventoryIds)
            ->where('status', 'reservado')
            ->get();
            
        $this->assertCount(3, $reserved);
        
        foreach ($reserved as $item) {
            $this->assertEquals('reservado', $item->status);
            $this->assertEquals($order->id, $item->order_id);
            $this->assertEquals('BOX001', $item->n_carton);
        }
    }

    /** @test */
    public function test_cannot_reserve_already_reserved_inventory()
    {
        // Create first draft order
        $order1 = Order::factory()->create([
            'client_id' => $this->client->id,
            'status' => 'draft',
        ]);
        
        // Reserve items
        $items = Inventory::where('n_carton', 'BOX001')->take(3)->get();
        foreach ($items as $item) {
            OrderItem::create([
                'order_id' => $order1->id,
                'inventory_id' => $item->id,
                'full_barcode' => $item->full_barcode,
                'mocaco' => $item->mocaco,
                'n_carton' => $item->n_carton,
            ]);
            
            $item->update([
                'status' => 'reservado',
                'order_id' => $order1->id,
            ]);
        }
        
        // Try to create another order with same boxes
        $response = $this->postJson('/api/order-preparation/save-draft', [
            'client_id' => $this->client->id,
            'operario_id' => 'OP002',
            'boxes' => ['BOX001'],
        ]);
        
        // Should only reserve items that are still 'disponible'
        $order2 = Order::latest()->first();
        $order2Items = OrderItem::where('order_id', $order2->id)->get();
        
        // Should create order but only reserve available items
        $this->assertNotNull($order2);
        // The exact count depends on how many items are still 'disponible'
    }
}

