<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Preparación de Pedidos - Sistema de Inventario</title>
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        :root {
            --sap-blue-dark: #2C3E50;
            --sap-blue-medium: #34495E;
            --sap-blue-light: #3498DB;
            --sap-background-light: #F8F9FB;
            --sap-content-background: #FFFFFF;
            --sap-text-dark: #333333;
            --sap-text-medium: #555555;
            --sap-border-color: #E0E0E0;
            --sap-shadow-light: rgba(0,0,0,0.08);
            --sap-shadow-medium: rgba(0,0,0,0.15);
        }

        body {
            font-family: 'Segoe UI', Arial, sans-serif;
            background-color: var(--sap-background-light);
            color: var(--sap-text-dark);
        }

        .form-control {
            border: 1px solid var(--sap-border-color);
            border-radius: 4px;
            padding: 10px;
            font-size: 0.9em;
            transition: border-color 0.2s, box-shadow 0.2s;
        }

        .form-control:focus {
            border-color: var(--sap-blue-light);
            outline: none;
            box-shadow: 0 0 0 2px rgba(52, 152, 219, 0.2);
        }

        .message {
            padding: 12px;
            margin-bottom: 15px;
            border-radius: 4px;
            font-weight: bold;
            display: block;
            text-align: center;
            font-size: 0.9em;
        }

        .message.success { 
            background-color: #d4edda; 
            color: #155724; 
            border: 1px solid #c3e6cb; 
        }

        .message.error { 
            background-color: #f8d7da; 
            color: #721c24; 
            border: 1px solid #f5c6cb; 
        }

        .message.info { 
            background-color: #d1ecf1; 
            color: #0c5460; 
            border: 1px solid #bee5eb; 
        }

        .box-item {
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 8px 12px;
            margin-bottom: 5px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .box-item .remove-box {
            color: #dc3545;
            cursor: pointer;
            font-size: 1.2em;
        }

        .box-item .remove-box:hover {
            color: #c82333;
        }

        .draft-item {
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 10px;
            margin-bottom: 8px;
        }

        .draft-item h6 {
            margin-bottom: 5px;
            color: #495057;
        }

        .draft-item small {
            color: #6c757d;
        }
    </style>
</head>
<body>
<div class="container-fluid mt-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-box-open"></i> Preparación y Envío de Pedido</h2>
        <a href="/test/clients" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Volver
        </a>
    </div>

    <div class="row">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-shopping-cart"></i> Crear Nuevo Pedido</h5>
                </div>
                <div class="card-body">
                    <form id="orderForm">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="client_select" class="form-label">Seleccionar Cliente *</label>
                                    <select id="client_select" name="client_select" class="form-control" required>
                                        <option value="">Cargando clientes...</option>
                                    </select>
                                    <div id="client_error" class="text-danger" style="display:none;"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="operarioId" class="form-label">ID del Operario *</label>
                                    <input type="text" id="operarioId" name="operario_id" class="form-control" 
                                           placeholder="Ej. OP001" required>
                                    <div id="operario_error" class="text-danger" style="display:none;"></div>
                                </div>
                            </div>
                        </div>

                        <div class="form-group mb-3">
                            <label for="order_notes" class="form-label">Notas del Pedido</label>
                            <textarea id="order_notes" name="order_notes" class="form-control" rows="2" 
                                      placeholder="Notas adicionales sobre el pedido o cliente"></textarea>
                        </div>
                        
                        <hr>

                        <div class="form-group mb-3">
                            <label for="barcodeInput" class="form-label">Escanee Código de Barras (N_CARTON) *</label>
                            <input type="text" id="barcodeInput" name="barcode_input" class="form-control" 
                                   placeholder="Escanee el número de caja" autofocus>
                            <div id="orderMessage" class="message mt-2" style="display:none;"></div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <h5>Cajas del Pedido Actual: <span id="boxCount" class="badge bg-primary">0</span></h5>
                            </div>
                            <div class="col-md-6 text-end">
                                <button type="button" id="reset_order_btn" class="btn btn-warning">
                                    <i class="fas fa-undo"></i> Reiniciar Pedido
                                </button>
                            </div>
                        </div>
                        
                        <div id="boxList" class="mt-3">
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle"></i> Escanee un número de caja para comenzar.
                            </div>
                        </div>

                        <div class="d-flex justify-content-end gap-2 mt-4">
                            <button type="submit" id="confirm_order_btn" class="btn btn-success" disabled>
                                <i class="fas fa-save"></i> Guardar Borrador de Pedido
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-md-4">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-info-circle"></i> Información</h5>
                </div>
                <div class="card-body">
                    <p class="text-muted">
                        <strong>Instrucciones:</strong>
                    </p>
                    <ol class="text-muted small">
                        <li>Seleccione un cliente de la lista</li>
                        <li>Ingrese el ID del operario</li>
                        <li>Escanee los números de caja</li>
                        <li>Agregue notas si es necesario</li>
                        <li>Guarde el borrador del pedido</li>
                    </ol>
                    
                    <hr>
                    
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i>
                        <strong>Importante:</strong> Los artículos serán eliminados del inventario cuando se confirme el despacho.
                    </div>
                </div>
            </div>

            <div class="card mt-3">
                <div class="card-header">
                    <h5><i class="fas fa-history"></i> Borradores Recientes</h5>
                </div>
                <div class="card-body">
                    <div id="recentDrafts">
                        <p class="text-muted text-center">Cargando borradores...</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const clientSelect = document.getElementById('client_select');
    const orderNotes = document.getElementById('order_notes');
    const operarioIdInput = document.getElementById('operarioId');
    const barcodeInput = document.getElementById('barcodeInput');
    const orderMessageDiv = document.getElementById('orderMessage');
    const boxListElement = document.getElementById('boxList');
    const boxCountSpan = document.getElementById('boxCount');
    const confirmOrderBtn = document.getElementById('confirm_order_btn');
    const resetOrderBtn = document.getElementById('reset_order_btn');
    const recentDraftsDiv = document.getElementById('recentDrafts');

    let scannedBoxes = new Set();

    // Initialize Select2
    $('#client_select').select2({
        placeholder: "Busque o seleccione un cliente",
        allowClear: true,
        width: '100%'
    });

    function showMessage(msg, type) {
        orderMessageDiv.textContent = msg;
        orderMessageDiv.className = `message ${type}`;
        orderMessageDiv.style.display = 'block';
        setTimeout(() => {
            orderMessageDiv.style.display = 'none';
        }, 5000);
    }

    function updateButtonState() {
        const hasBoxes = scannedBoxes.size > 0;
        const hasClient = clientSelect.value;
        const hasOperario = operarioIdInput.value.trim();
        
        confirmOrderBtn.disabled = !hasBoxes || !hasClient || !hasOperario;
        
        if (!hasBoxes) {
            confirmOrderBtn.innerHTML = '<i class="fas fa-save"></i> Guardar Borrador de Pedido';
        } else {
            confirmOrderBtn.innerHTML = `<i class="fas fa-save"></i> Guardar Borrador (${scannedBoxes.size} cajas)`;
        }
    }

    async function loadClients() {
        try {
            const response = await fetch('/api/order-preparation/clients');
            const clients = await response.json();
            
            $('#client_select').empty().append($('<option value="">Seleccione un cliente...</option>')).trigger('change');

            if (clients.length > 0) {
                clients.forEach(client => {
                    const newOption = new Option(client.client_name, client.client_id, false, false);
                    $('#client_select').append(newOption);
                });
                $('#client_select').trigger('change');
            } else {
                $('#client_select').append($('<option value="">No hay clientes disponibles.</option>')).trigger('change');
            }
        } catch (error) {
            console.error('Error al cargar clientes:', error);
            showMessage('Error de red al cargar clientes.', 'error');
        }
    }

    function addBoxToList(boxNumber) {
        if (scannedBoxes.has(boxNumber)) {
            showMessage(`La caja ${boxNumber} ya ha sido añadida a este pedido.`, 'error');
            return;
        }

        scannedBoxes.add(boxNumber);
        updateBoxList();
        boxCountSpan.textContent = scannedBoxes.size;
        showMessage(`Caja ${boxNumber} añadida al pedido.`, 'success');
        updateButtonState();
    }

    function removeBoxFromList(boxNumber) {
        scannedBoxes.delete(boxNumber);
        updateBoxList();
        boxCountSpan.textContent = scannedBoxes.size;
        showMessage(`Caja ${boxNumber} eliminada del pedido.`, 'info');
        updateButtonState();
    }

    function updateBoxList() {
        if (scannedBoxes.size === 0) {
            boxListElement.innerHTML = '<div class="alert alert-info"><i class="fas fa-info-circle"></i> Escanee un número de caja para comenzar.</div>';
        } else {
            let html = '';
            scannedBoxes.forEach(boxNumber => {
                html += `
                    <div class="box-item">
                        <span><i class="fas fa-box"></i> ${boxNumber}</span>
                        <span class="remove-box" onclick="removeBoxFromList('${boxNumber}')" title="Eliminar caja">
                            <i class="fas fa-times"></i>
                        </span>
                    </div>
                `;
            });
            boxListElement.innerHTML = html;
        }
    }

    // Event listeners
    barcodeInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            const boxNumber = e.target.value.trim();
            if (boxNumber) {
                addBoxToList(boxNumber);
                e.target.value = '';
            }
        }
    });

    clientSelect.addEventListener('change', updateButtonState);
    operarioIdInput.addEventListener('input', updateButtonState);

    document.getElementById('orderForm').addEventListener('submit', async (e) => {
        e.preventDefault();
        showMessage('Guardando borrador...', 'info');

        const formData = {
            client_id: clientSelect.value,
            operario_id: operarioIdInput.value,
            notes: orderNotes.value,
            boxes: Array.from(scannedBoxes)
        };

        if (formData.boxes.length === 0) {
            showMessage('No se han agregado cajas.', 'error');
            return;
        }

        confirmOrderBtn.disabled = true;
        confirmOrderBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Guardando...';

        try {
            const response = await fetch('/api/order-preparation/save-draft', {
                method: 'POST',
                headers: { 
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify(formData)
            });
            
            const data = await response.json();

            if (data.success) {
                showMessage(`Borrador de pedido #${data.order_id} guardado con éxito.`, 'success');
                // Clear form
                $('#client_select').val('').trigger('change');
                orderNotes.value = '';
                operarioIdInput.value = '';
                scannedBoxes.clear();
                updateBoxList();
                boxCountSpan.textContent = 0;
                loadClients();
                loadRecentDrafts();
            } else {
                showMessage(`Error: ${data.message}`, 'error');
            }
        } catch (error) {
            console.error('Error en la solicitud:', error);
            showMessage(`Error en la solicitud: ${error.message}`, 'error');
        } finally {
            updateButtonState();
        }
    });

    resetOrderBtn.addEventListener('click', () => {
        $('#client_select').val('').trigger('change');
        orderNotes.value = '';
        operarioIdInput.value = '';
        scannedBoxes.clear();
        updateBoxList();
        boxCountSpan.textContent = 0;
        showMessage('', 'info');
        updateButtonState();
        loadClients();
    });

    async function loadRecentDrafts() {
        try {
            const response = await fetch('/api/order-preparation/draft-orders');
            const drafts = await response.json();
            
            if (drafts.length > 0) {
                let html = '';
                drafts.slice(0, 5).forEach(draft => {
                    const date = new Date(draft.order_date).toLocaleDateString();
                    html += `
                        <div class="draft-item">
                            <h6>Pedido #${draft.id}</h6>
                            <small>Cliente: ${draft.client.client_name}</small><br>
                            <small>Fecha: ${date}</small><br>
                            <small>Operario: ${draft.operario_id}</small>
                        </div>
                    `;
                });
                recentDraftsDiv.innerHTML = html;
            } else {
                recentDraftsDiv.innerHTML = '<p class="text-muted text-center">No hay borradores recientes.</p>';
            }
        } catch (error) {
            console.error('Error al cargar borradores:', error);
            recentDraftsDiv.innerHTML = '<p class="text-muted text-center">Error al cargar borradores.</p>';
        }
    }

    // Initialize
    loadClients();
    loadRecentDrafts();
    updateButtonState();
});
</script>
</body>
</html>
