<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Asignación de Ubicación - Sistema de Inventario</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        :root {
            --sap-blue-dark: #2C3E50;
            --sap-blue-medium: #34495E;
            --sap-blue-light: #3498DB;
            --sap-background-light: #F8F9FB;
            --sap-content-background: #FFFFFF;
            --sap-text-dark: #333333;
            --sap-text-medium: #555555;
            --sap-border-color: #E0E0E0;
            --sap-shadow-light: rgba(0,0,0,0.08);
            --sap-shadow-medium: rgba(0,0,0,0.15);
        }

        body {
            font-family: 'Segoe UI', Arial, sans-serif;
            background-color: var(--sap-background-light);
            color: var(--sap-text-dark);
        }

        .form-control {
            border: 1px solid var(--sap-border-color);
            border-radius: 4px;
            padding: 10px;
            font-size: 0.9em;
            transition: border-color 0.2s, box-shadow 0.2s;
        }

        .form-control:focus {
            border-color: var(--sap-blue-light);
            outline: none;
            box-shadow: 0 0 0 2px rgba(52, 152, 219, 0.2);
        }

        .message {
            padding: 12px;
            margin-bottom: 15px;
            border-radius: 4px;
            font-weight: bold;
            display: block;
            text-align: center;
            font-size: 0.9em;
        }

        .message.success { 
            background-color: #d4edda; 
            color: #155724; 
            border: 1px solid #c3e6cb; 
        }

        .message.error { 
            background-color: #f8d7da; 
            color: #721c24; 
            border: 1px solid #f5c6cb; 
        }

        .message.info { 
            background-color: #d1ecf1; 
            color: #0c5460; 
            border: 1px solid #bee5eb; 
        }

        .box-item {
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 8px 12px;
            margin-bottom: 5px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .box-item .remove-box {
            color: #dc3545;
            cursor: pointer;
            font-size: 1.2em;
        }

        .box-item .remove-box:hover {
            color: #c82333;
        }

        .location-item {
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 10px;
            margin-bottom: 8px;
        }

        .location-item h6 {
            margin-bottom: 5px;
            color: #495057;
        }

        .location-item small {
            color: #6c757d;
        }

        .stats-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }

        .stats-icon {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--sap-blue-light), #2980b9);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 15px;
            color: white;
            font-size: 1.5em;
        }

        .stats-content h3 {
            font-size: 2em;
            font-weight: bold;
            color: var(--sap-blue-dark);
            margin: 0;
            line-height: 1;
        }

        .stats-content p {
            color: var(--sap-text-medium);
            margin: 5px 0 0 0;
            font-size: 0.9em;
        }
    </style>
</head>
<body>
<div class="container-fluid mt-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-map-marker-alt"></i> Asignación de Ubicación por Caja</h2>
        <a href="/test/clients" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Volver
        </a>
    </div>

    <div class="row">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-map-marker-alt"></i> Asignar Ubicación</h5>
                </div>
                <div class="card-body">
                    <form id="locationForm">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="ubicacion_input" class="form-label">Ubicación *</label>
                                    <input type="text" id="ubicacion_input" name="ubicacion_input" class="form-control" 
                                           placeholder="Ej. 01-02-01" required>
                                    <div id="ubicacion_error" class="text-danger" style="display:none;"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group mb-3">
                                    <label for="operarioId" class="form-label">Nº ID Operario *</label>
                                    <input type="text" id="operarioId" name="operario_id" class="form-control" 
                                           placeholder="Ej. OP001" value="OP001" required>
                                    <div id="operario_error" class="text-danger" style="display:none;"></div>
                                </div>
                            </div>
                        </div>
                        
                        <hr>

                        <div class="form-group mb-3">
                            <label for="barcode_caja_input" class="form-label">Escanee Nº Caja (N_CARTON) *</label>
                            <input type="text" id="barcode_caja_input" name="barcode_caja_input" class="form-control" 
                                   placeholder="Escanee el número de caja" autofocus>
                            <div id="message" class="message mt-2" style="display:none;"></div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <h5>Cajas Escaneadas: <span id="boxCount" class="badge bg-primary">0</span></h5>
                            </div>
                            <div class="col-md-6 text-end">
                                <button type="button" id="reset_button" class="btn btn-warning">
                                    <i class="fas fa-undo"></i> Limpiar Todo
                                </button>
                            </div>
                        </div>
                        
                        <div id="scanned_boxes_list" class="mt-3">
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle"></i> Escanee un número de caja para comenzar.
                            </div>
                        </div>

                        <div class="d-flex justify-content-end gap-2 mt-4">
                            <button type="submit" id="confirm_button" class="btn btn-success" disabled>
                                <i class="fas fa-check"></i> Confirmar Asignación
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-md-4">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-info-circle"></i> Información</h5>
                </div>
                <div class="card-body">
                    <p class="text-muted">
                        <strong>Instrucciones:</strong>
                    </p>
                    <ol class="text-muted small">
                        <li>Ingrese la ubicación deseada</li>
                        <li>Verifique el ID del operario</li>
                        <li>Escanee los números de caja</li>
                        <li>Confirme la asignación</li>
                    </ol>
                    
                    <hr>
                    
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i>
                        <strong>Formato de Ubicación:</strong> Use el formato XX-XX-XX (ej: 01-02-01)
                    </div>
                </div>
            </div>

            <div class="card mt-3">
                <div class="card-header">
                    <h5><i class="fas fa-chart-bar"></i> Estadísticas</h5>
                </div>
                <div class="card-body">
                    <div class="stats-card d-flex align-items-center">
                        <div class="stats-icon">
                            <i class="fas fa-boxes"></i>
                        </div>
                        <div class="stats-content">
                            <h3 id="totalItems">-</h3>
                            <p>Items en Inventario</p>
                        </div>
                    </div>
                    
                    <div class="stats-card d-flex align-items-center">
                        <div class="stats-icon">
                            <i class="fas fa-map-marker-alt"></i>
                        </div>
                        <div class="stats-content">
                            <h3 id="totalLocations">-</h3>
                            <p>Ubicaciones Únicas</p>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card mt-3">
                <div class="card-header">
                    <h5><i class="fas fa-history"></i> Ubicaciones Recientes</h5>
                </div>
                <div class="card-body">
                    <div id="recentLocations">
                        <p class="text-muted text-center">Cargando ubicaciones...</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const ubicacionInput = document.getElementById('ubicacion_input');
    const operarioIdInput = document.getElementById('operarioId');
    const barcodeCajaInput = document.getElementById('barcode_caja_input');
    const messageDiv = document.getElementById('message');
    const scannedBoxesList = document.getElementById('scanned_boxes_list');
    const confirmButton = document.getElementById('confirm_button');
    const resetButton = document.getElementById('reset_button');
    const boxCountSpan = document.getElementById('boxCount');
    const recentLocationsDiv = document.getElementById('recentLocations');
    const totalItemsSpan = document.getElementById('totalItems');
    const totalLocationsSpan = document.getElementById('totalLocations');

    let scannedBoxes = new Set();

    function showMessage(msg, type) {
        messageDiv.textContent = msg;
        messageDiv.className = `message ${type}`;
        messageDiv.style.display = 'block';
        setTimeout(() => {
            messageDiv.style.display = 'none';
        }, 5000);
    }

    function updateButtonState() {
        const hasBoxes = scannedBoxes.size > 0;
        const hasUbicacion = ubicacionInput.value.trim();
        const hasOperario = operarioIdInput.value.trim();
        
        confirmButton.disabled = !hasBoxes || !hasUbicacion || !hasOperario;
        
        if (!hasBoxes) {
            confirmButton.innerHTML = '<i class="fas fa-check"></i> Confirmar Asignación';
        } else {
            confirmButton.innerHTML = `<i class="fas fa-check"></i> Confirmar Asignación (${scannedBoxes.size} cajas)`;
        }
    }

    function addBoxToList(boxNumber) {
        if (scannedBoxes.has(boxNumber)) {
            showMessage(`La caja ${boxNumber} ya ha sido añadida.`, 'error');
            return;
        }

        scannedBoxes.add(boxNumber);
        updateBoxList();
        boxCountSpan.textContent = scannedBoxes.size;
        showMessage(`Caja ${boxNumber} añadida.`, 'success');
        updateButtonState();
    }

    function removeBoxFromList(boxNumber) {
        scannedBoxes.delete(boxNumber);
        updateBoxList();
        boxCountSpan.textContent = scannedBoxes.size;
        showMessage(`Caja ${boxNumber} eliminada.`, 'info');
        updateButtonState();
    }

    function updateBoxList() {
        if (scannedBoxes.size === 0) {
            scannedBoxesList.innerHTML = '<div class="alert alert-info"><i class="fas fa-info-circle"></i> Escanee un número de caja para comenzar.</div>';
        } else {
            let html = '';
            scannedBoxes.forEach(boxNumber => {
                html += `
                    <div class="box-item">
                        <span><i class="fas fa-box"></i> ${boxNumber}</span>
                        <span class="remove-box" onclick="removeBoxFromList('${boxNumber}')" title="Eliminar caja">
                            <i class="fas fa-times"></i>
                        </span>
                    </div>
                `;
            });
            scannedBoxesList.innerHTML = html;
        }
    }

    // Event listeners
    barcodeCajaInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            const boxNumber = e.target.value.trim();
            if (boxNumber) {
                addBoxToList(boxNumber);
                e.target.value = '';
            }
        }
    });

    ubicacionInput.addEventListener('input', updateButtonState);
    operarioIdInput.addEventListener('input', updateButtonState);

    document.getElementById('locationForm').addEventListener('submit', async (e) => {
        e.preventDefault();
        showMessage('Asignando ubicación...', 'info');

        const formData = {
            ubicacion: ubicacionInput.value.trim(),
            operario_id: operarioIdInput.value.trim(),
            boxes: Array.from(scannedBoxes)
        };

        if (formData.boxes.length === 0) {
            showMessage('No se han agregado cajas.', 'error');
            return;
        }

        confirmButton.disabled = true;
        confirmButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Asignando...';

        try {
            const response = await fetch('/api/location/assign', {
                method: 'POST',
                headers: { 
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify(formData)
            });
            
            const data = await response.json();

            if (data.success) {
                showMessage(data.message, 'success');
                // Clear form
                scannedBoxes.clear();
                updateBoxList();
                boxCountSpan.textContent = 0;
                loadStatistics();
                loadRecentLocations();
            } else {
                showMessage(`Error: ${data.message}`, 'error');
            }
        } catch (error) {
            console.error('Error en la solicitud:', error);
            showMessage(`Error en la solicitud: ${error.message}`, 'error');
        } finally {
            updateButtonState();
        }
    });

    resetButton.addEventListener('click', () => {
        scannedBoxes.clear();
        updateBoxList();
        boxCountSpan.textContent = 0;
        showMessage('', 'info');
        updateButtonState();
    });

    async function loadStatistics() {
        try {
            // Load total items count
            const itemsResponse = await fetch('/api/inventory/statistics');
            const itemsData = await itemsResponse.json();
            totalItemsSpan.textContent = itemsData.total_items || 0;

            // Load total locations count
            const locationsResponse = await fetch('/api/location/locations');
            const locations = await locationsResponse.json();
            totalLocationsSpan.textContent = locations.length || 0;
        } catch (error) {
            console.error('Error al cargar estadísticas:', error);
            totalItemsSpan.textContent = '-';
            totalLocationsSpan.textContent = '-';
        }
    }

    async function loadRecentLocations() {
        try {
            const response = await fetch('/api/location/locations');
            const locations = await response.json();
            
            if (locations.length > 0) {
                let html = '';
                locations.slice(0, 5).forEach(location => {
                    html += `
                        <div class="location-item">
                            <h6><i class="fas fa-map-marker-alt"></i> ${location}</h6>
                            <small>Ubicación registrada</small>
                        </div>
                    `;
                });
                recentLocationsDiv.innerHTML = html;
            } else {
                recentLocationsDiv.innerHTML = '<p class="text-muted text-center">No hay ubicaciones registradas.</p>';
            }
        } catch (error) {
            console.error('Error al cargar ubicaciones:', error);
            recentLocationsDiv.innerHTML = '<p class="text-muted text-center">Error al cargar ubicaciones.</p>';
        }
    }

    // Initialize
    loadStatistics();
    loadRecentLocations();
    updateButtonState();
});
</script>
</body>
</html>
