<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Historial de Clientes - Sistema de Inventario</title>
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        :root {
            --sap-blue-dark: #2C3E50;
            --sap-blue-medium: #34495E;
            --sap-blue-light: #3498DB;
            --sap-background-light: #F8F9FB;
            --sap-content-background: #FFFFFF;
            --sap-text-dark: #333333;
            --sap-text-medium: #555555;
            --sap-border-color: #E0E0E0;
            --sap-shadow-light: rgba(0,0,0,0.08);
            --sap-shadow-medium: rgba(0,0,0,0.15);
        }

        body {
            font-family: 'Segoe UI', Arial, sans-serif;
            background-color: var(--sap-background-light);
            color: var(--sap-text-dark);
        }

        .form-control {
            border: 1px solid var(--sap-border-color);
            border-radius: 4px;
            padding: 10px;
            font-size: 0.9em;
            transition: border-color 0.2s, box-shadow 0.2s;
        }

        .form-control:focus {
            border-color: var(--sap-blue-light);
            outline: none;
            box-shadow: 0 0 0 2px rgba(52, 152, 219, 0.2);
        }

        .message {
            padding: 12px;
            margin-bottom: 15px;
            border-radius: 4px;
            font-weight: bold;
            display: block;
            text-align: center;
            font-size: 0.9em;
        }

        .message.success { 
            background-color: #d4edda; 
            color: #155724; 
            border: 1px solid #c3e6cb; 
        }

        .message.error { 
            background-color: #f8d7da; 
            color: #721c24; 
            border: 1px solid #f5c6cb; 
        }

        .message.info { 
            background-color: #d1ecf1; 
            color: #0c5460; 
            border: 1px solid #bee5eb; 
        }

        .order-item {
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 15px;
            margin-bottom: 15px;
            transition: all 0.2s ease;
        }

        .order-item:hover {
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .order-item h6 {
            margin-bottom: 8px;
            color: #495057;
        }

        .order-item small {
            color: #6c757d;
        }

        .stats-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }

        .stats-icon {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--sap-blue-light), #2980b9);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 15px;
            color: white;
            font-size: 1.5em;
        }

        .stats-content h3 {
            font-size: 2em;
            font-weight: bold;
            color: var(--sap-blue-dark);
            margin: 0;
            line-height: 1;
        }

        .stats-content p {
            color: var(--sap-text-medium);
            margin: 5px 0 0 0;
            font-size: 0.9em;
        }

        .table-responsive {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .table {
            margin-bottom: 0;
        }

        .table th {
            background-color: #f8f9fa;
            border-top: none;
            font-weight: 600;
            color: var(--sap-text-medium);
            font-size: 0.85em;
            padding: 12px;
        }

        .table td {
            padding: 12px;
            font-size: 0.85em;
            vertical-align: middle;
            border-top: 1px solid #dee2e6;
        }

        .table tbody tr:hover {
            background-color: #f8f9fa;
        }

        .badge {
            padding: 4px 8px;
            font-size: 0.75em;
            font-weight: 500;
            border-radius: 4px;
        }

        .badge-success {
            background-color: #28a745;
            color: white;
        }

        .btn-sm {
            padding: 5px 10px;
            font-size: 0.8em;
            min-width: auto;
        }

        .client-info {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }

        .client-info h4 {
            margin-bottom: 10px;
        }

        .client-info p {
            margin-bottom: 5px;
            opacity: 0.9;
        }
    </style>
</head>
<body>
<div class="container-fluid mt-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-history"></i> Historial de Clientes y Pedidos Enviados</h2>
        <a href="/test/clients" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Volver
        </a>
    </div>

    <div class="row">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-handshake"></i> Seleccionar Cliente</h5>
                </div>
                <div class="card-body">
                    <div class="form-group mb-3">
                        <label for="client_history_select" class="form-label">Seleccionar Cliente *</label>
                        <select id="client_history_select" name="client_history_select" class="form-control" required>
                            <option value="">Cargando clientes...</option>
                        </select>
                        <div id="client_error" class="text-danger" style="display:none;"></div>
                    </div>
                </div>
            </div>

            <div id="client_info_section" class="client-info" style="display:none;">
                <h4 id="client_name_display"></h4>
                <p id="client_contact_display"></p>
                <p id="client_email_display"></p>
                <p id="client_phone_display"></p>
            </div>

            <div id="order_history_results" class="mt-4">
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> Seleccione un cliente para ver su historial de pedidos.
                </div>
            </div>
        </div>

        <div class="col-md-4">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-info-circle"></i> Información</h5>
                </div>
                <div class="card-body">
                    <p class="text-muted">
                        <strong>Instrucciones:</strong>
                    </p>
                    <ol class="text-muted small">
                        <li>Seleccione un cliente de la lista</li>
                        <li>Visualice su historial de pedidos</li>
                        <li>Consulte detalles de cada pedido</li>
                        <li>Exporte el historial si es necesario</li>
                    </ol>
                    
                    <hr>
                    
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i>
                        <strong>Nota:</strong> Solo se muestran pedidos con estado "despachado".
                    </div>
                </div>
            </div>

            <div class="card mt-3">
                <div class="card-header">
                    <h5><i class="fas fa-chart-bar"></i> Estadísticas del Cliente</h5>
                </div>
                <div class="card-body">
                    <div id="client_stats">
                        <p class="text-muted text-center">Seleccione un cliente para ver estadísticas.</p>
                    </div>
                </div>
            </div>

            <div class="card mt-3">
                <div class="card-header">
                    <h5><i class="fas fa-download"></i> Exportar Historial</h5>
                </div>
                <div class="card-body">
                    <div id="export_section">
                        <p class="text-muted text-center">Seleccione un cliente para exportar su historial.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const clientHistorySelect = document.getElementById('client_history_select');
    const orderHistoryResults = document.getElementById('order_history_results');
    const clientInfoSection = document.getElementById('client_info_section');
    const clientStatsDiv = document.getElementById('client_stats');
    const exportSectionDiv = document.getElementById('export_section');

    // Initialize Select2
    $('#client_history_select').select2({
        placeholder: "Busque o seleccione un cliente",
        allowClear: true,
        width: '100%'
    });

    function showMessage(msg, type) {
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${type}`;
        messageDiv.textContent = msg;
        messageDiv.style.display = 'block';
        
        orderHistoryResults.innerHTML = '';
        orderHistoryResults.appendChild(messageDiv);
        
        setTimeout(() => {
            messageDiv.style.display = 'none';
        }, 5000);
    }

    async function loadClients() {
        try {
            const response = await fetch('/api/client-history/clients');
            const clients = await response.json();
            
            $('#client_history_select').empty().append($('<option value="">Seleccione un cliente...</option>')).trigger('change');

            if (clients.length > 0) {
                clients.forEach(client => {
                    const newOption = new Option(client.client_name, client.client_id, false, false);
                    $('#client_history_select').append(newOption);
                });
                $('#client_history_select').trigger('change');
            } else {
                $('#client_history_select').append($('<option value="">No hay clientes disponibles.</option>')).trigger('change');
                showMessage('No se pudieron cargar los clientes.', 'error');
            }
        } catch (error) {
            console.error('Error al cargar clientes:', error);
            showMessage('Error de red al cargar clientes.', 'error');
        }
    }

    async function loadClientOrders(clientId) {
        try {
            showMessage('Cargando historial de pedidos...', 'info');
            
            const response = await fetch(`/api/client-history/orders/${clientId}`);
            const orders = await response.json();

            if (orders.length > 0) {
                let html = '<h5><i class="fas fa-list-alt"></i> Historial de Pedidos</h5>';
                
                orders.forEach(order => {
                    const orderDate = new Date(order.order_date).toLocaleDateString();
                    const dispatchDate = order.dispatched_date ? new Date(order.dispatched_date).toLocaleDateString() : 'N/A';
                    const itemsCount = order.order_items ? order.order_items.length : 0;
                    
                    html += `
                        <div class="order-item">
                            <div class="d-flex justify-content-between align-items-start">
                                <div>
                                    <h6><i class="fas fa-shopping-cart"></i> Pedido #${order.id}</h6>
                                    <small><strong>Fecha Pedido:</strong> ${orderDate}</small><br>
                                    <small><strong>Fecha Despacho:</strong> ${dispatchDate}</small><br>
                                    <small><strong>Operario:</strong> ${order.operario_id}</small><br>
                                    <small><strong>Artículos:</strong> ${itemsCount}</small>
                                    ${order.notes ? `<br><small><strong>Notas:</strong> ${order.notes}</small>` : ''}
                                </div>
                                <div>
                                    <button class="btn btn-sm btn-outline-primary" onclick="showOrderDetails(${order.id})">
                                        <i class="fas fa-eye"></i> Ver Detalles
                                    </button>
                                </div>
                            </div>
                        </div>
                    `;
                });
                
                orderHistoryResults.innerHTML = html;
            } else {
                orderHistoryResults.innerHTML = `
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i> No se encontraron pedidos despachados para este cliente.
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error al cargar pedidos:', error);
            showMessage('Error al cargar el historial de pedidos.', 'error');
        }
    }

    async function loadClientStats(clientId) {
        try {
            const response = await fetch(`/api/client-history/stats/${clientId}`);
            const stats = await response.json();
            
            const lastOrderDate = stats.last_order_date ? new Date(stats.last_order_date).toLocaleDateString() : 'N/A';
            
            clientStatsDiv.innerHTML = `
                <div class="stats-card d-flex align-items-center">
                    <div class="stats-icon">
                        <i class="fas fa-shopping-cart"></i>
                    </div>
                    <div class="stats-content">
                        <h3>${stats.total_orders}</h3>
                        <p>Pedidos Totales</p>
                    </div>
                </div>
                
                <div class="stats-card d-flex align-items-center">
                    <div class="stats-icon">
                        <i class="fas fa-boxes"></i>
                    </div>
                    <div class="stats-content">
                        <h3>${stats.total_items}</h3>
                        <p>Artículos Despachados</p>
                    </div>
                </div>
                
                <div class="stats-card d-flex align-items-center">
                    <div class="stats-icon">
                        <i class="fas fa-archive"></i>
                    </div>
                    <div class="stats-content">
                        <h3>${stats.total_boxes}</h3>
                        <p>Cajas Únicas</p>
                    </div>
                </div>
                
                <div class="alert alert-info">
                    <small><strong>Último Pedido:</strong> ${lastOrderDate}</small>
                </div>
            `;
        } catch (error) {
            console.error('Error al cargar estadísticas:', error);
            clientStatsDiv.innerHTML = '<p class="text-muted text-center">Error al cargar estadísticas.</p>';
        }
    }

    async function loadClientInfo(clientId) {
        try {
            const response = await fetch(`/api/client-history/stats/${clientId}`);
            const stats = await response.json();
            
            const client = stats.client_info;
            
            document.getElementById('client_name_display').textContent = client.client_name;
            document.getElementById('client_contact_display').textContent = `Contacto: ${client.contact_person || 'N/A'}`;
            document.getElementById('client_email_display').textContent = `Email: ${client.email || 'N/A'}`;
            document.getElementById('client_phone_display').textContent = `Teléfono: ${client.phone || 'N/A'}`;
            
            clientInfoSection.style.display = 'block';
            
            // Update export section
            exportSectionDiv.innerHTML = `
                <button class="btn btn-success w-100" onclick="exportClientHistory(${clientId})">
                    <i class="fas fa-download"></i> Exportar Historial CSV
                </button>
            `;
        } catch (error) {
            console.error('Error al cargar información del cliente:', error);
        }
    }

    // Event listener for client selection
    clientHistorySelect.addEventListener('change', async function() {
        const clientId = this.value;
        
        if (clientId) {
            await Promise.all([
                loadClientOrders(clientId),
                loadClientStats(clientId),
                loadClientInfo(clientId)
            ]);
        } else {
            orderHistoryResults.innerHTML = `
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> Seleccione un cliente para ver su historial de pedidos.
                </div>
            `;
            clientInfoSection.style.display = 'none';
            clientStatsDiv.innerHTML = '<p class="text-muted text-center">Seleccione un cliente para ver estadísticas.</p>';
            exportSectionDiv.innerHTML = '<p class="text-muted text-center">Seleccione un cliente para exportar su historial.</p>';
        }
    });

    // Global functions for order details and export
    window.showOrderDetails = async function(orderId) {
        try {
            const response = await fetch(`/api/client-history/order-details/${orderId}`);
            const order = await response.json();
            
            let itemsHtml = '';
            if (order.order_items && order.order_items.length > 0) {
                itemsHtml = `
                    <div class="table-responsive mt-3">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Código de Barras</th>
                                    <th>MOCACO</th>
                                    <th>Número de Caja</th>
                                </tr>
                            </thead>
                            <tbody>
                `;
                
                order.order_items.forEach(item => {
                    itemsHtml += `
                        <tr>
                            <td><code>${item.full_barcode}</code></td>
                            <td>${item.mocaco}</td>
                            <td>${item.n_carton}</td>
                        </tr>
                    `;
                });
                
                itemsHtml += `
                            </tbody>
                        </table>
                    </div>
                `;
            }
            
            const modalHtml = `
                <div class="modal fade" id="orderDetailsModal" tabindex="-1">
                    <div class="modal-dialog modal-lg">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">Detalles del Pedido #${order.id}</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <p><strong>Cliente:</strong> ${order.client.client_name}</p>
                                        <p><strong>Fecha Pedido:</strong> ${new Date(order.order_date).toLocaleDateString()}</p>
                                        <p><strong>Fecha Despacho:</strong> ${order.dispatched_date ? new Date(order.dispatched_date).toLocaleDateString() : 'N/A'}</p>
                                        <p><strong>Operario:</strong> ${order.operario_id}</p>
                                    </div>
                                    <div class="col-md-6">
                                        <p><strong>Estado:</strong> <span class="badge badge-success">Despachado</span></p>
                                        <p><strong>Total Artículos:</strong> ${order.order_items ? order.order_items.length : 0}</p>
                                        ${order.notes ? `<p><strong>Notas:</strong> ${order.notes}</p>` : ''}
                                    </div>
                                </div>
                                ${itemsHtml}
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            // Remove existing modal if any
            const existingModal = document.getElementById('orderDetailsModal');
            if (existingModal) {
                existingModal.remove();
            }
            
            // Add modal to body
            document.body.insertAdjacentHTML('beforeend', modalHtml);
            
            // Show modal
            const modal = new bootstrap.Modal(document.getElementById('orderDetailsModal'));
            modal.show();
        } catch (error) {
            console.error('Error al cargar detalles del pedido:', error);
            showMessage('Error al cargar los detalles del pedido.', 'error');
        }
    };

    window.exportClientHistory = function(clientId) {
        window.open(`/api/client-history/export/${clientId}`, '_blank');
    };

    // Initialize
    loadClients();
});
</script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
