@extends('layouts.app')

@section('title', __('relabeling.title'))

@section('content')
<div class="container">
    <h1>{{ __('relabeling.title') }}</h1>

    <div id="messageDiv" class="message" style="display: none;">
        <span id="messageText"></span>
    </div>

    <!-- Modo de Escaneo -->
    <div id="scanModeSection">
        <form id="scanForm" method="post">
            @csrf
            <div class="form-group">
                <label for="scanned_code">{{ __('relabeling.scan_mode.scan_barcode') }}</label>
                <input type="text" id="scanned_code" name="scanned_code" required>
            </div>
            <button type="submit" style="display:none;"></button>
        </form>

        <div id="printer_controls" style="margin-top:20px;">
            <button id="checkConnectionBtn" class="btn-check">{{ __('relabeling.scan_mode.check_connection') }}</button>
            <button id="testPrintBtn" class="btn-check">{{ __('relabeling.scan_mode.test_print') }}</button>
            <button id="retryPrintBtn" class="btn-check" style="display:none;">{{ __('relabeling.scan_mode.retry_print') }}</button>
        </div>

        <div id="printerMessage" class="message" style="display:none; margin-top:10px;"></div>
    </div>

    <!-- Modo de Edición Manual (producto no encontrado) -->
    <div id="manualModeSection" style="display:none;">
        <form id="manualForm" method="post">
            @csrf
            <div class="message error">
                <strong>{{ __('relabeling.manual_mode.product_not_found') }}</strong> {{ __('relabeling.manual_mode.complete_data') }}
            </div>
            <input type="hidden" name="proveedor_code" id="manual_proveedor_code">
            <div class="form-group">
                <label for="codebar_gnu">{{ __('relabeling.fields.codebar_gnu_label') }}</label>
                <input type="text" id="codebar_gnu" name="codebar_gnu" required>
            </div>
            <div class="form-group">
                <label for="description">{{ __('relabeling.fields.description') }}:</label>
                <input type="text" id="description" name="description" required>
            </div>
            <div class="form-group">
                <label for="price_cfa">{{ __('relabeling.fields.price_cfa_label') }}</label>
                <input type="text" id="price_cfa" name="price_cfa" required>
            </div>
            <button type="submit" name="save_and_print" class="btn-submit">{{ __('relabeling.buttons.save_and_print') }}</button>
        </form>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/qz-tray@latest"></script>
<script src="{{ asset('js/qz-tray-config.js') }}"></script>
@php
$relabelingTranslations = [
    'product_found' => __('relabeling.messages.product_found'),
    'product_not_found' => __('relabeling.messages.product_not_found'),
    'communication_error' => __('relabeling.messages.communication_error'),
    'product_saved' => __('relabeling.messages.product_saved'),
    'save_error' => __('relabeling.messages.save_error'),
    'save_communication_error' => __('relabeling.messages.save_communication_error'),
    'connection_active' => __('relabeling.messages.connection_active'),
    'connection_success' => __('relabeling.messages.connection_success'),
    'connection_error' => __('relabeling.messages.connection_error'),
    'print_success' => __('relabeling.messages.print_success'),
    'print_error' => __('relabeling.messages.print_error'),
    'code_int' => __('relabeling.labels.code_int'),
];
@endphp
<script>
// Inyectar traducciones en JavaScript
const translations = JSON.parse('{!! addslashes(json_encode($relabelingTranslations)) !!}');

// Función helper para formatear mensajes con parámetros
function formatMessage(key, params = {}) {
    let message = translations[key] || key;
    Object.keys(params).forEach(param => {
        message = message.replace(`:${param}`, params[param]);
    });
    return message;
}

const scannedInput = document.getElementById('scanned_code');
const messageDiv = document.getElementById('messageDiv');
const messageText = document.getElementById('messageText');
const scanModeSection = document.getElementById('scanModeSection');
const manualModeSection = document.getElementById('manualModeSection');
const scanForm = document.getElementById('scanForm');
const manualForm = document.getElementById('manualForm');
const printerMsg = document.getElementById('printerMessage');
const retryBtn = document.getElementById('retryPrintBtn');

let labelData = {};
let printNow = false;

// Auto-focus en el campo de escaneo
scannedInput.focus();
scannedInput.addEventListener('keydown', function(event) {
    if(event.key === 'Enter') {
        event.preventDefault();
        handleScan();
    }
});

// Función para mostrar mensajes
function showMessage(msg, type = 'success') {
    messageText.textContent = msg;
    messageDiv.className = 'message ' + type;
    messageDiv.style.display = 'block';
}

// Función para manejar el escaneo
async function handleScan() {
    const scannedCode = scannedInput.value.trim();
    if (!scannedCode) return;

    try {
        const response = await fetch('/relabeling/search-product', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({ proveedor_code: scannedCode })
        });

        const data = await response.json();

        if (data.success && data.product) {
            // Producto encontrado
            labelData = {
                barcode: data.product.proveedor_code,
                my_code: data.product.codebar_gnu,
                desc: data.product.description,
                price: data.product.price_cfa || '0'
            };
            showMessage(translations.product_found, 'success');
            printNow = true;
            
            // Imprimir automáticamente
            setTimeout(() => sendPrint(labelData), 500);
            
            // Limpiar input
            scannedInput.value = '';
            scannedInput.focus();
        } else {
            // Producto no encontrado
            manualModeSection.style.display = 'block';
            scanModeSection.style.display = 'none';
            document.getElementById('manual_proveedor_code').value = scannedCode;
            showMessage(translations.product_not_found, 'error');
        }
    } catch (error) {
        console.error('Error al buscar producto:', error);
        showMessage(translations.communication_error, 'error');
    }
}

// Submit del formulario de escaneo
scanForm.addEventListener('submit', function(e) {
    e.preventDefault();
    handleScan();
});

// Submit del formulario manual
manualForm.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = {
        proveedor_code: document.getElementById('manual_proveedor_code').value,
        codebar_gnu: document.getElementById('codebar_gnu').value,
        description: document.getElementById('description').value,
        price_cfa: document.getElementById('price_cfa').value
    };

    try {
        const response = await fetch('/relabeling/save-product', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(formData)
        });

        const data = await response.json();

        if (data.success) {
            labelData = {
                barcode: formData.proveedor_code,
                my_code: formData.codebar_gnu,
                desc: formData.description,
                price: formData.price_cfa
            };
            showMessage(translations.product_saved, 'success');
            
            // Imprimir automáticamente
            setTimeout(() => sendPrint(labelData), 500);
            
            // Volver al modo de escaneo
            manualModeSection.style.display = 'none';
            scanModeSection.style.display = 'block';
            scannedInput.focus();
            
            // Limpiar formulario manual
            manualForm.reset();
        } else {
            showMessage(formatMessage('save_error', { message: data.message || '' }), 'error');
        }
    } catch (error) {
        console.error('Error al guardar producto:', error);
        showMessage(translations.save_communication_error, 'error');
    }
});

// Función para mostrar mensaje de impresora
function showMsg(msg, type = 'success') {
    printerMsg.textContent = msg;
    printerMsg.className = 'message ' + type;
    printerMsg.style.display = 'block';
}

// Función para conectar y verificar el estado de QZ Tray (usando la configuración con firma)
async function checkConnection() {
    try {
        if (qz.websocket.isActive()) {
            showMsg(translations.connection_active, 'success');
            return true;
        }
        // Usar la función de conexión con firma personalizada
        if (typeof connectQZTray === 'function') {
            await connectQZTray();
        } else {
            // Fallback si no está cargada la configuración
        await qz.api.connect();
        }
        showMsg(translations.connection_success, 'success');
        return true;
    } catch(err) {
        showMsg(formatMessage('connection_error', { message: err.message }), 'error');
        return false;
    }
}

// Función para generar el código ZPL
function getZplContent(data) {
    let zpl = '^XA';
    
    // Campo 1: Descripción
    zpl += '^FO20,20^ADN,36,20^FD' + data.desc + '^FS';
    
    // Campo 2: Código Interno GNU
    zpl += '^FO20,65^ADN,18,10^FD' + translations.code_int + ' ' + data.my_code + '^FS';
    
    // Campo 3: Precio (más grande)
    zpl += '^FO20,100^A0N,60,60^FDCFA ' + data.price + '^FS';
    
    // Campo 4: Código de Barras (PROVEEDOR CODE)
    zpl += '^FO20,170^BY3^BCN,100,Y,N,N^FD' + data.barcode + '^FS';
    
    zpl += '^XZ';
    return zpl;
}

// Función para enviar la impresión
async function sendPrint(data) {
    if (!await checkConnection()) {
        retryBtn.style.display = 'inline-block';
        return;
    }

    try {
        const printerName = await qz.printers.getDefault();
        
        const config = qz.configs.create(printerName, {
            jobName: "Reetiquetado - " + data.barcode,
            forceRaw: true,
            copies: 1
        });
        
        const content = getZplContent(data);
        
        await qz.print(config, [content]);
        
        showMsg(formatMessage('print_success', { barcode: data.barcode, printer: printerName }), 'success');
        retryBtn.style.display = 'none';

    } catch(err) {
        showMsg(formatMessage('print_error', { message: err.message }), 'error');
        retryBtn.style.display = 'inline-block';
    }
}

// Eventos de botones
document.getElementById('checkConnectionBtn').addEventListener('click', checkConnection);

document.getElementById('testPrintBtn').addEventListener('click', () => sendPrint({
    barcode: 'TEST12345',
    my_code: '123456',
    desc: 'PRUEBA DE CONEXIÓN',
    price: '999'
}));

retryBtn.addEventListener('click', () => sendPrint(labelData));
</script>
@endsection

@push('styles')
<style>
.container {
    max-width: 800px;
    margin: 0 auto;
    padding: 20px;
}

.message {
    padding: 10px;
    margin-bottom: 15px;
    border-radius: 4px;
}

.message.success {
    background-color: #d4edda;
    color: #155724;
}

.message.error {
    background-color: #f8d7da;
    color: #721c24;
}

.btn-check {
    margin-right: 10px;
}

.form-group {
    margin-bottom: 15px;
}

.form-group label {
    display: block;
    margin-bottom: 5px;
    font-weight: bold;
}

.form-group input {
    width: 100%;
    padding: 8px;
    border: 1px solid #ddd;
    border-radius: 4px;
}

.btn-submit {
    background-color: var(--sap-blue-light);
    color: white;
    padding: 10px 20px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
}

.btn-submit:hover {
    background-color: var(--sap-blue-dark);
}
</style>
@endpush

