@extends('layouts.app')

@section('title', __('order_preparation.title'))

@section('content')
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2><i class="fas fa-box-open"></i> {{ __('order_preparation.subtitle') }}</h2>
    <div class="d-flex gap-2">
        <a href="{{ route('reservations.index') }}" class="btn btn-outline-primary">
            <i class="fas fa-clipboard-list"></i> Ver Todas las Reservas
        </a>
        <a href="{{ route('dashboard') }}" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> {{ __('order_preparation.back') }}
        </a>
    </div>
</div>

<!-- Sistema de Pestañas -->
<ul class="nav nav-tabs mb-4" id="orderPrepTabs" role="tablist">
    <!-- Tab 1: Nuevo Pedido (Scan to Order) -->
    <li class="nav-item" role="presentation">
        <button class="nav-link active" id="new-order-tab" data-bs-toggle="tab" data-bs-target="#new-order" type="button" role="tab" aria-controls="new-order" aria-selected="true">
            <i class="fas fa-plus"></i> {{ __('order_preparation.create_order') }}
        </button>
    </li>

    <!-- Tab 2: Reservas de Cajas (NUEVO - Box Reservations) -->
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="box-reservations-tab" data-bs-toggle="tab" data-bs-target="#box-reservations" type="button" role="tab" aria-controls="box-reservations" aria-selected="false">
            <i class="fas fa-boxes"></i> Reservas de Cajas
            <span id="pending-reservations-badge" class="badge bg-danger rounded-pill d-none">0</span>
        </button>
    </li>

    <!-- Tab 3: Pedidos Reservados (Antiguo confirmación de pedidos) -->
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="reserved-orders-tab" data-bs-toggle="tab" data-bs-target="#reserved-orders" type="button" role="tab" aria-controls="reserved-orders" aria-selected="false">
            <i class="fas fa-clipboard-check"></i> Pedidos Reservados
        </button>
    </li>
</ul>

<div class="tab-content" id="orderPrepTabsContent">

    <!-- CONTENIDO TAB 1: NUEVO PEDIDO -->
    <div class="tab-pane fade show active" id="new-order" role="tabpanel" aria-labelledby="new-order-tab">
        <div class="row">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-shopping-cart"></i> {{ __('order_preparation.create_order') }}</h5>
                    </div>
                    <div class="card-body">
                        <form id="orderForm">
                            @csrf

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group mb-3">
                                        <label for="client_select" class="form-label">{{ __('order_preparation.select_client') }}</label>
                                        <select id="client_select" name="client_select" class="form-control" required>
                                            <option value="">{{ __('order_preparation.loading_clients') }}</option>
                                        </select>
                                        <div id="client_error" class="text-danger" style="display:none;"></div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group mb-3">
                                        <label for="operarioId" class="form-label">{{ __('order_preparation.operator_id') }}</label>
                                        <input type="text" id="operarioId" name="operario_id" class="form-control"
                                               value="{{ $operarioId ?? '' }}"
                                               placeholder="{{ __('order_preparation.operator_placeholder') }}" required>
                                        <div id="operario_error" class="text-danger" style="display:none;"></div>
                                    </div>
                                </div>
                            </div>

                            <div class="form-group mb-3">
                                <label for="order_notes" class="form-label">{{ __('order_preparation.order_notes') }}</label>
                                <textarea id="order_notes" name="order_notes" class="form-control" rows="2"
                                          placeholder="{{ __('order_preparation.order_notes_placeholder') }}"></textarea>
                            </div>

                            <hr>

                            <div class="form-group mb-3">
                                <label for="barcodeInput" class="form-label">{{ __('order_preparation.scan_barcode') }}</label>
                                <input type="text" id="barcodeInput" name="barcode_input" class="form-control"
                                       placeholder="{{ __('order_preparation.scan_barcode_placeholder') }}">
                                <div id="orderMessage" class="message mt-2" style="display:none;"></div>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <h5>{{ __('order_preparation.current_boxes') }} <span id="boxCount" class="badge bg-primary">0</span></h5>
                                </div>
                                <div class="col-md-6 text-end">
                                    <button type="button" id="reset_order_btn" class="btn btn-warning">
                                        <i class="fas fa-undo"></i> {{ __('order_preparation.reset_order') }}
                                    </button>
                                </div>
                            </div>

                            <div id="boxList" class="mt-3">
                                <div class="alert alert-info">
                                    <i class="fas fa-info-circle"></i> {{ __('order_preparation.scan_to_start') }}
                                </div>
                            </div>

                            <div class="d-flex justify-content-end gap-2 mt-4">
                                <button type="submit" id="confirm_order_btn" class="btn btn-success" disabled>
                                    <i class="fas fa-save"></i> {{ __('order_preparation.save_draft') }}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <div class="col-md-4">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-info-circle"></i> {{ __('order_preparation.info') }}</h5>
                    </div>
                    <div class="card-body">
                        <p class="text-muted">
                            <strong>{{ __('order_preparation.instructions') }}</strong>
                        </p>
                        <ol class="text-muted small">
                            <li>{{ __('order_preparation.instruction_1') }}</li>
                            <li>{{ __('order_preparation.instruction_2') }}</li>
                            <li>{{ __('order_preparation.instruction_3') }}</li>
                            <li>{{ __('order_preparation.instruction_4') }}</li>
                            <li>{{ __('order_preparation.instruction_5') }}</li>
                        </ol>

                        <hr>

                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i>
                            <strong>{{ __('order_preparation.important') }}</strong> {{ __('order_preparation.important_note') }}
                        </div>
                    </div>
                </div>

                <div class="card mt-3">
                    <div class="card-header">
                        <h5><i class="fas fa-history"></i> {{ __('order_preparation.recent_drafts') }}</h5>
                    </div>
                    <div class="card-body">
                        <div id="recentDrafts">
                            <p class="text-muted text-center">{{ __('order_preparation.loading_drafts') }}</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- CONTENIDO TAB 2: RESERVAS DE CAJAS (BOX RESERVATIONS Implementation) -->
    <div class="tab-pane fade" id="box-reservations" role="tabpanel" aria-labelledby="box-reservations-tab">
        <div class="card mb-4 shadow-sm">
            <div class="card-header bg-info text-white">
                <h5 class="mb-0"><i class="fas fa-boxes"></i> Gestión de Reservas de Cajas</h5>
            </div>
            <div class="card-body">
                <!-- Selector de Reservas -->
                <div class="row mb-4">
                    <div class="col-md-8">
                        <label for="box_reservation_select" class="form-label">Seleccionar Reserva Activa</label>
                        <select id="box_reservation_select" class="form-control">
                            <option value="">Cargando reservas...</option>
                        </select>
                    </div>
                    <div class="col-md-4 d-flex align-items-end">
                        <button type="button" id="load_box_reservation_btn" class="btn btn-primary w-100" disabled>
                            <i class="fas fa-play"></i> Iniciar/Continuar Reserva
                        </button>
                    </div>
                </div>

                <!-- Área de Trabajo de Reserva -->
                <div id="box_reservation_workspace" style="display: none;">
                    <!-- Información de la Reserva -->
                    <div class="alert alert-light border shadow-sm">
                        <div class="row">
                            <div class="col-md-6">
                                <h6>
                                    <strong>Cliente:</strong> <span id="br_client_name" class="fw-bold"></span>
                                    <button class="btn btn-sm btn-link p-0 ms-2" id="br_edit_details_btn" title="Editar detalles">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                </h6>
                                <p class="mb-0 text-muted">ID Reserva: #<span id="br_id"></span> | <span id="br_status_badge"></span></p>
                            </div>
                            <div class="col-md-6 text-end">
                                <div class="h4 mb-0">
                                    <span id="br_progress_percent">0</span>%
                                </div>
                                <div class="progress" style="height: 10px;">
                                    <div id="br_progress_bar" class="progress-bar bg-success" role="progressbar" style="width: 0%"></div>
                                </div>
                                <small>Confirmadas: <span id="br_confirmed_count">0</span> / <span id="br_total_count">0</span></small>
                            </div>
                        </div>
                        <div class="mt-2 text-muted small border-top pt-2">
                            <i class="fas fa-sticky-note"></i> <strong>Notas:</strong> <span id="br_notes">Sin notas</span>
                        </div>
                    </div>

                    <div class="row">
                        <!-- Columna Izquierda: Input y Listas -->
                        <div class="col-md-6">
                            <!-- Input Principal -->
                            <div class="card mb-3 border-primary">
                                <div class="card-header bg-primary text-white">
                                    <h6 class="mb-0"><i class="fas fa-barcode"></i> Escanear para Confirmar</h6>
                                </div>
                                <div class="card-body bg-light">
                                    <input type="text" id="br_barcode_input" class="form-control form-control-lg text-center fw-bold" placeholder="ESCANEAR CÓDIGO CAJA" autocomplete="off">
                                    <div id="br_message" class="mt-2 text-center fw-bold p-2 rounded" style="display:none;"></div>
                                </div>
                            </div>

                            <!-- Última Acción -->
                            <div id="br_last_action" class="alert alert-secondary py-2 mb-3" style="display: none;">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-history me-2"></i>
                                    <div>
                                        <small class="fw-bold d-block">Último escaneo:</small>
                                        <span id="br_last_action_text"></span>
                                    </div>
                                </div>
                            </div>

                            <!-- Lista de Pendientes (Nuevo Requerimiento) -->
                            <div id="br_pending_list_container" class="mb-3">
                                <!-- La tabla se inyecta desde JS -->
                            </div>

                             <!-- Cajas Extra -->
                            <div class="card mb-3 border-warning">
                                <div class="card-header bg-warning-subtle text-warning-emphasis py-2 d-flex justify-content-between align-items-center">
                                    <h6 class="mb-0 font-size-sm"><i class="fas fa-plus-circle"></i> Cajas Extra (No en reserva)</h6>
                                    <span class="badge bg-warning text-dark" id="br_extra_count">0</span>
                                </div>
                                <div class="card-body p-2">
                                     <div class="input-group mb-2">
                                        <input type="text" id="br_extra_input" class="form-control form-control-sm" placeholder="Escanear caja extra...">
                                        <button class="btn btn-outline-warning" type="button" id="br_add_extra_btn">Añadir</button>
                                    </div>
                                    <div id="br_extra_list" class="d-flex flex-wrap gap-2" style="max-height: 100px; overflow-y: auto;">
                                        <!-- Badges de cajas extra -->
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Columna Derecha: Listado de Cajas -->
                        <div class="col-md-6">
                            <div class="card h-100">
                                <div class="card-header py-2 d-flex justify-content-between align-items-center">
                                    <h6 class="mb-0">Contenido de la Reserva</h6>
                                    <div class="btn-group btn-group-sm">
                                        <button type="button" class="btn btn-outline-primary active btn-filter-br" data-filter="all">Todas</button>
                                        <button type="button" class="btn btn-outline-primary btn-filter-br" data-filter="pending">Pendientes</button>
                                        <button type="button" class="btn btn-outline-primary btn-filter-br" data-filter="confirmed">Listas</button>
                                    </div>
                                </div>
                                <div class="card-body p-0" style="height: 400px; overflow-y: auto; background-color: #f8f9fa;">
                                    <div id="br_boxes_list_container" class="list-group list-group-flush">
                                        <!-- Items generados dinámicamente -->
                                        <div class="text-center p-4 text-muted">
                                            <i class="fas fa-spinner fa-spin"></i> Cargando items...
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Botones de Acción -->
                    <div class="d-flex justify-content-between align-items-center mt-3 pt-3 border-top">
                        <div>
                            <button type="button" class="btn btn-outline-secondary" onclick="location.reload()">
                                <i class="fas fa-sign-out-alt"></i> Salir (Guardar Progreso)
                            </button>
                            <button type="button" id="br_cancel_btn" class="btn btn-outline-danger ms-2">
                                <i class="fas fa-times-circle"></i> Cancelar Reserva
                            </button>
                        </div>
                        <div>
                            <span class="text-muted me-2 small" id="br_completion_status">Faltan cajas para completar</span>
                            <button type="button" id="br_complete_btn" class="btn btn-success btn-lg" disabled>
                                <i class="fas fa-check-double"></i> Completar y Generar Pedido
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Edit Details Modal -->
        <div class="modal fade" id="editReservationModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Editar Detalles de Reserva</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <input type="hidden" id="edit_br_id">
                        <div class="mb-3">
                            <label for="edit_br_client" class="form-label">Cliente</label>
                            <select id="edit_br_client" class="form-control" style="width: 100%"></select>
                        </div>
                        <div class="mb-3">
                            <label for="edit_br_notes" class="form-label">Notas</label>
                            <textarea id="edit_br_notes" class="form-control" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="button" class="btn btn-primary" id="save_br_details_btn">Guardar Cambios</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- CONTENIDO TAB 3: PEDIDOS RESERVADOS (Existing implementation) -->
    <div class="tab-pane fade" id="reserved-orders" role="tabpanel" aria-labelledby="reserved-orders-tab">
        <div class="card mb-4 shadow-sm">
            <div class="card-header bg-warning text-dark">
                <h5 class="mb-0"><i class="fas fa-clipboard-check"></i> Confirmar Pedidos Reservados (Status: Reserved)</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <label for="reserved_order_select" class="form-label">Seleccionar Pedido Reservado</label>
                        <select id="reserved_order_select" class="form-control">
                            <option value="">Cargando pedidos reservados...</option>
                        </select>
                    </div>
                    <div class="col-md-6 d-flex align-items-end">
                        <button type="button" id="load_reserved_order_btn" class="btn btn-primary" disabled>
                            <i class="fas fa-download"></i> Cargar Pedido
                        </button>
                    </div>
                </div>

                <div id="reserved_order_info" class="mt-3" style="display: none;">
                    <div class="alert alert-info">
                        <h6>Pedido: <span id="reserved_order_id"></span> - Cliente: <span id="reserved_order_client"></span></h6>
                        <p class="mb-0">Cajas reservadas: <strong id="reserved_boxes_count">0</strong> | Total items: <strong id="reserved_items_count">0</strong></p>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Cajas Reservadas (a confirmar):</label>
                        <div id="reserved_boxes_list" class="border p-2 rounded" style="max-height: 150px; overflow-y: auto; background-color: #f8f9fa;">
                            <!-- Se llenará dinámicamente -->
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="reserved_barcode_input" class="form-label">Escanear Código de Barras para Confirmar:</label>
                        <input type="text" id="reserved_barcode_input" class="form-control" placeholder="Escanea las cajas físicamente...">
                        <div id="reserved_order_message" class="message mt-2" style="display:none;"></div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Cajas Escaneadas (Confirmadas):</label>
                        <div id="scanned_boxes_list" class="border p-2 rounded" style="max-height: 150px; overflow-y: auto; background-color: #e7f3ff;">
                            <p class="text-muted text-center mb-0">Aún no se han escaneado cajas</p>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="extra_boxes_input" class="form-label">Cajas Extra (opcional - escanear para añadir):</label>
                        <input type="text" id="extra_boxes_input" class="form-control" placeholder="Escanea cajas adicionales si es necesario...">
                        <small class="text-muted">Puedes añadir cajas extra que no estaban en la reserva (ej: un pallet completo)</small>
                    </div>

                    <div id="extra_boxes_list" class="mb-3" style="display: none;">
                        <label class="form-label">Cajas Extra Añadidas:</label>
                        <div class="border p-2 rounded" style="max-height: 100px; overflow-y: auto; background-color: #fff3cd;">
                            <!-- Se llenará dinámicamente -->
                        </div>
                    </div>

                    <div class="d-flex justify-content-end gap-2">
                        <button type="button" id="cancel_reserved_order_btn" class="btn btn-secondary">
                            <i class="fas fa-times"></i> Cancelar
                        </button>
                        <button type="button" id="confirm_reserved_order_btn" class="btn btn-success" disabled>
                            <i class="fas fa-check"></i> Confirmar y Despachar Pedido
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<style>
.form-control {
    border: 1px solid var(--sap-border-color);
    border-radius: 4px;
    padding: 10px;
    font-size: 0.9em;
    transition: border-color 0.2s, box-shadow 0.2s;
}

.form-control:focus {
    border-color: var(--sap-blue-light);
    outline: none;
    box-shadow: 0 0 0 2px rgba(52, 152, 219, 0.2);
}

.message {
    padding: 12px;
    margin-bottom: 15px;
    border-radius: 4px;
    font-weight: bold;
    display: block;
    text-align: center;
    font-size: 0.9em;
}

.message.success {
    background-color: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background-color: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.message.info {
    background-color: #d1ecf1;
    color: #0c5460;
    border: 1px solid #bee5eb;
}

.box-item {
    background-color: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 4px;
    padding: 8px 12px;
    margin-bottom: 5px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.box-item .remove-box {
    color: #dc3545;
    cursor: pointer;
    font-size: 1.2em;
}

.box-item .remove-box:hover {
    color: #c82333;
}

.draft-item {
    background-color: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 4px;
    padding: 10px;
    margin-bottom: 8px;
}

.draft-item h6 {
    margin-bottom: 5px;
    color: #495057;
}

.draft-item small {
    color: #6c757d;
}

.gap-2 {
    gap: 10px;
}

/* Styles for Box Reservation */
.br-box-item {
    padding: 10px;
    border-bottom: 1px solid #dee2e6;
    transition: background-color 0.2s;
}
.br-box-item:hover {
    background-color: #f1f3f5;
}
.br-box-item.confirmed {
    background-color: #d4edda;
    border-left: 4px solid #28a745;
}
.br-box-item.pending {
    border-left: 4px solid #ffc107;
}
.br-box-status {
    font-size: 0.8em;
    font-weight: bold;
    text-transform: uppercase;
}
</style>
@endpush

@push('scripts')
@php
$orderPreparationTranslations = [
    'select_client_placeholder' => __('order_preparation.select_client_placeholder'),
    'select_client_option' => __('order_preparation.select_client_option'),
    'no_clients' => __('order_preparation.no_clients'),
    'scan_to_start' => __('order_preparation.scan_to_start'),
    'save_draft' => __('order_preparation.save_draft'),
    'save_draft_with_boxes' => __('order_preparation.save_draft_with_boxes'),
    'saving' => __('order_preparation.saving'),
    'box_already_added' => __('order_preparation.messages.box_already_added'),
    'box_not_exists' => __('order_preparation.messages.box_not_exists'),
    'box_added' => __('order_preparation.messages.box_added'),
    'box_removed' => __('order_preparation.messages.box_removed'),
    'no_boxes' => __('order_preparation.messages.no_boxes'),
    'saving_draft' => __('order_preparation.messages.saving_draft'),
    'draft_saved' => __('order_preparation.messages.draft_saved'),
    'error' => __('order_preparation.messages.error'),
    'request_error' => __('order_preparation.messages.request_error'),
    'network_error' => __('order_preparation.messages.network_error'),
    'validation_error' => __('order_preparation.messages.validation_error'),
    'remove_box_title' => __('order_preparation.remove_box_title'),
    'loading_drafts' => __('order_preparation.loading_drafts'),
    'no_recent_drafts' => __('order_preparation.no_recent_drafts'),
    'error_loading_drafts' => __('order_preparation.error_loading_drafts'),
    'order' => __('order_preparation.order'),
    'client' => __('order_preparation.client'),
    'date' => __('order_preparation.date'),
    'operator' => __('order_preparation.operator'),
];
@endphp

<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
// Inyectar traducciones en JavaScript
const translations = JSON.parse('{!! addslashes(json_encode($orderPreparationTranslations)) !!}');

// Función helper para formatear mensajes con parámetros
function formatMessage(key, params = {}) {
    let message = translations[key] || key;
    Object.keys(params).forEach(param => {
        message = message.replace(`:${param}`, params[param]);
    });
    return message;
}

document.addEventListener('DOMContentLoaded', function() {

    // ==========================================
    // LOGICA TAB 1: NUEVO PEDIDO
    // ==========================================
    const clientSelect = document.getElementById('client_select');
    const orderNotes = document.getElementById('order_notes');
    const operarioIdInput = document.getElementById('operarioId');
    const barcodeInput = document.getElementById('barcodeInput');
    const orderMessageDiv = document.getElementById('orderMessage');
    const boxListElement = document.getElementById('boxList');
    const boxCountSpan = document.getElementById('boxCount');
    const confirmOrderBtn = document.getElementById('confirm_order_btn');
    const resetOrderBtn = document.getElementById('reset_order_btn');
    const recentDraftsDiv = document.getElementById('recentDrafts');

    let scannedBoxes = new Set();

    // Initialize Select2
    $('#client_select').select2({
        placeholder: translations.select_client_placeholder,
        allowClear: true,
        width: '100%'
    });

    function showMessage(msg, type) {
        orderMessageDiv.textContent = msg;
        orderMessageDiv.className = `message ${type}`;
        orderMessageDiv.style.display = 'block';
        setTimeout(() => {
            orderMessageDiv.style.display = 'none';
        }, 5000);
    }

    function updateButtonState() {
        const hasBoxes = scannedBoxes.size > 0;
        const hasClient = clientSelect.value;
        const hasOperario = operarioIdInput.value.trim();

        confirmOrderBtn.disabled = !hasBoxes || !hasClient || !hasOperario;

        if (!hasBoxes) {
            confirmOrderBtn.innerHTML = `<i class="fas fa-save"></i> ${translations.save_draft}`;
        } else {
            confirmOrderBtn.innerHTML = `<i class="fas fa-save"></i> ${formatMessage('save_draft_with_boxes', { count: scannedBoxes.size })}`;
        }
    }

    async function loadClients() {
        try {
            const response = await fetch('/api/order-preparation/clients');
            const clients = await response.json();

            $('#client_select').empty().append($(`<option value="">${translations.select_client_option}</option>`)).trigger('change');

            if (clients.length > 0) {
                clients.forEach(client => {
                    const newOption = new Option(client.client_name, client.client_id, false, false);
                    $('#client_select').append(newOption);
                });
                $('#client_select').trigger('change');
            } else {
                $('#client_select').append($(`<option value="">${translations.no_clients}</option>`)).trigger('change');
            }
        } catch (error) {
            console.error('Error al cargar clientes:', error);
            showMessage(translations.network_error, 'error');
        }
    }

    async function addBoxToList(boxNumber) {
        if (scannedBoxes.has(boxNumber)) {
            showMessage(formatMessage('box_already_added', { box: boxNumber }), 'error');
            return;
        }

        try {
            const response = await fetch('/api/order-preparation/validate-box', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({ input: boxNumber })
            });

            if (!response.ok) throw new Error(`HTTP: ${response.status}`);

            const data = await response.json();

            if (!data.exists) {
                showMessage(formatMessage('box_not_exists', { box: boxNumber }), 'error');
                return;
            }

            scannedBoxes.add(boxNumber);
            updateBoxList();
            boxCountSpan.textContent = scannedBoxes.size;
            showMessage(formatMessage('box_added', { box: boxNumber, count: data.count }), 'success');
            updateButtonState();
        } catch (error) {
            console.error('Error:', error);
            showMessage(formatMessage('validation_error', { message: error.message }), 'error');
        }
    }

    function removeBoxFromList(boxNumber) {
        scannedBoxes.delete(boxNumber);
        updateBoxList();
        boxCountSpan.textContent = scannedBoxes.size;
        showMessage(formatMessage('box_removed', { box: boxNumber }), 'info');
        updateButtonState();
    }

    function updateBoxList() {
        if (scannedBoxes.size === 0) {
            boxListElement.innerHTML = `<div class="alert alert-info"><i class="fas fa-info-circle"></i> ${translations.scan_to_start}</div>`;
        } else {
            let html = '';
            scannedBoxes.forEach(boxNumber => {
                html += `
                    <div class="box-item">
                        <span><i class="fas fa-box"></i> ${boxNumber}</span>
                        <span class="remove-box" data-box="${boxNumber.replace(/"/g, '&quot;')}" title="${translations.remove_box_title}">
                            <i class="fas fa-times"></i>
                        </span>
                    </div>
                `;
            });
            boxListElement.innerHTML = html;
        }
    }

    barcodeInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            const boxNumber = e.target.value.trim();
            if (boxNumber) {
                addBoxToList(boxNumber);
                e.target.value = '';
            }
        }
    });

    boxListElement.addEventListener('click', (e) => {
        if (e.target.closest('.remove-box')) {
            const boxNumber = e.target.closest('.remove-box').getAttribute('data-box');
            if (boxNumber) removeBoxFromList(boxNumber);
        }
    });

    clientSelect.addEventListener('change', updateButtonState);
    operarioIdInput.addEventListener('input', updateButtonState);
    operarioIdInput.addEventListener('change', updateButtonState);

    document.getElementById('orderForm').addEventListener('submit', async (e) => {
        e.preventDefault();
        showMessage(translations.saving_draft, 'info');

        const formData = {
            client_id: clientSelect.value,
            operario_id: operarioIdInput.value,
            notes: orderNotes.value,
            boxes: Array.from(scannedBoxes)
        };

        if (formData.boxes.length === 0) {
            showMessage(translations.no_boxes, 'error');
            return;
        }

        confirmOrderBtn.disabled = true;
        confirmOrderBtn.innerHTML = `<i class="fas fa-spinner fa-spin"></i> ${translations.saving}`;

        try {
            const response = await fetch('/api/order-preparation/save-draft', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify(formData)
            });

            const data = await response.json();

            if (data.success) {
                showMessage(formatMessage('draft_saved', { id: data.order_id }), 'success');
                $('#client_select').val('').trigger('change');
                orderNotes.value = '';
                operarioIdInput.value = '';
                scannedBoxes.clear();
                updateBoxList();
                boxCountSpan.textContent = 0;
                loadClients();
                loadRecentDrafts();
            } else {
                showMessage(formatMessage('error', { message: data.message }), 'error');
            }
        } catch (error) {
            console.error('Error:', error);
            showMessage(formatMessage('request_error', { message: error.message }), 'error');
        } finally {
            updateButtonState();
        }
    });

    resetOrderBtn.addEventListener('click', () => {
        $('#client_select').val('').trigger('change');
        orderNotes.value = '';
        operarioIdInput.value = '';
        scannedBoxes.clear();
        updateBoxList();
        boxCountSpan.textContent = 0;
        showMessage('', 'info');
        updateButtonState();
        loadClients();
    });

    async function loadRecentDrafts() {
        try {
            const response = await fetch('/api/order-preparation/draft-orders');
            const drafts = await response.json();

            if (drafts.length > 0) {
                let html = '';
                drafts.slice(0, 5).forEach(draft => {
                    const date = new Date(draft.order_date).toLocaleDateString();
                    html += `
                        <div class="draft-item">
                            <h6>${translations.order} #${draft.id}</h6>
                            <small>${translations.client} ${draft.client.client_name}</small><br>
                            <small>${translations.date} ${date}</small><br>
                            <small>${translations.operator} ${draft.operario_id}</small>
                        </div>
                    `;
                });
                recentDraftsDiv.innerHTML = html;
            } else {
                recentDraftsDiv.innerHTML = `<p class="text-muted text-center">${translations.no_recent_drafts}</p>`;
            }
        } catch (error) {
            recentDraftsDiv.innerHTML = `<p class="text-muted text-center">${translations.error_loading_drafts}</p>`;
        }
    }

    loadClients();
    loadRecentDrafts();
    updateButtonState();


    // ==========================================
    // LÓGICA TAB 2: RESERVAS DE CAJAS (NUEVO)
    // ==========================================

    // Elementos DOM
    const brSelect = document.getElementById('box_reservation_select');
    const brLoadBtn = document.getElementById('load_box_reservation_btn');
    const brWorkspace = document.getElementById('box_reservation_workspace');
    const brInput = document.getElementById('br_barcode_input');
    const brExtraInput = document.getElementById('br_extra_input');
    const brAddExtraBtn = document.getElementById('br_add_extra_btn');
    const brCompleteBtn = document.getElementById('br_complete_btn');
    const brCancelBtn = document.getElementById('br_cancel_btn');
    const brMessageDiv = document.getElementById('br_message');
    const brBoxListContainer = document.getElementById('br_boxes_list_container');
    const brPendingBadge = document.getElementById('pending-reservations-badge');

    let currentRes = null;
    let brBoxes = []; // Array of box objects from API
    let brExtraBoxes = new Set();
    let brFilter = 'all';

    async function loadBoxReservations() {
        try {
            const response = await fetch('/api/order-preparation/box-reservations');
            const data = await response.json();

            brSelect.innerHTML = '<option value="">Seleccionar una reserva pendiente...</option>';

            if(data.length > 0) {
                data.forEach(res => {
                    const date = new Date(res.created_at).toLocaleDateString();
                    const client = res.client_name || 'Sin cliente';
                    const option = document.createElement('option');
                    option.value = res.id;
                    option.textContent = `${res.reservation_code || res.id} - ${client} (${res.total_boxes} cajas) - ${date}`;
                    brSelect.appendChild(option);
                });
                brPendingBadge.textContent = data.length;
                brPendingBadge.classList.remove('d-none');
            } else {
                const option = document.createElement('option');
                option.text = "No hay reservas pendientes";
                brSelect.appendChild(option);
                brPendingBadge.classList.add('d-none');
            }
        } catch (error) {
            console.error(error);
            brSelect.innerHTML = '<option value="">Error al cargar</option>';
        }
    }

    function showBrMessage(msg, type) {
        brMessageDiv.textContent = msg;
        brMessageDiv.className = `mt-2 text-center fw-bold p-2 rounde alert alert-${type === 'error' ? 'danger' : type}`;
        brMessageDiv.style.display = 'block';
        setTimeout(() => { brMessageDiv.style.display = 'none'; }, 4000);
    }

    async function startReservation() {
        const id = brSelect.value;
        if (!id) return;

        brLoadBtn.disabled = true;
        brLoadBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Cargando...';

        try {
            const response = await fetch('/api/order-preparation/start-box-reservation', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({ reservation_id: id })
            });
            const data = await response.json();

            if (data.success) {
                currentRes = data.reservation;
                brBoxes = data.boxes;
                brExtraBoxes = new Set(data.extra_boxes || []);

                // Render UI
                document.getElementById('br_client_name').textContent = currentRes.client ? currentRes.client.client_name : 'No asignado';
                document.getElementById('br_id').textContent = currentRes.reservation_code || currentRes.id;
                document.getElementById('br_notes').textContent = currentRes.notes || 'Sin notas';
                document.getElementById('br_status_badge').innerHTML = `<span class="badge bg-primary">${currentRes.status}</span>`;

                renderBrState();

                brWorkspace.style.display = 'block';
                brInput.focus();
            } else {
                alert('No se pudo cargar la reserva: ' + data.message);
            }
        } catch (error) {
            console.error(error);
            alert('Error de conexión');
        } finally {
            brLoadBtn.disabled = false;
            brLoadBtn.innerHTML = '<i class="fas fa-play"></i> Iniciar/Continuar Reserva';
        }
    }

    function renderBrState() {
        // Calculate stats
        const confirmed = brBoxes.filter(b => b.status === 'confirmed').length;
        const total = brBoxes.length; // Original boxes

        // Progress bar
        const percent = total > 0 ? Math.round((confirmed / total) * 100) : 0;
        document.getElementById('br_progress_bar').style.width = `${percent}%`;
        document.getElementById('br_progress_percent').textContent = percent;
        document.getElementById('br_confirmed_count').textContent = confirmed;
        document.getElementById('br_total_count').textContent = total;

        // Limpiar contenedor duplicado de la izquierda (si existe) para evitar redundancia
        const pendingContainer = document.getElementById('br_pending_list_container');
        if(pendingContainer) pendingContainer.innerHTML = '';

        // Determine boxes to show based on filter
        let boxesToShow = brBoxes;
        if (brFilter === 'pending') {
            boxesToShow = brBoxes.filter(b => b.status !== 'confirmed');
        } else if (brFilter === 'confirmed') {
            boxesToShow = brBoxes.filter(b => b.status === 'confirmed');
        }

        // Render List in Right Column (brBoxListContainer)
        brBoxListContainer.innerHTML = '';

        if (boxesToShow.length > 0) {
            const table = document.createElement('table');
            table.className = 'table table-hover mb-0';
            table.innerHTML = `
                <thead class="table-light sticky-top">
                    <tr>
                        <th>Ubicación 📍</th>
                        <th>Nº Caja</th>
                        <th>Unidades</th>
                        <th>Estado</th>
                    </tr>
                </thead>
                <tbody></tbody>
            `;
            const tbody = table.querySelector('tbody');

            boxesToShow.forEach(box => {
                const tr = document.createElement('tr');
                const isConfirmed = box.status === 'confirmed';
                const ubicacion = box.inventory?.ubicacion || '-';

                tr.innerHTML = `
                    <td>${ubicacion}</td>
                    <td>#${box.box_number || '-'}</td>
                    <td>${box.quantity || '-'} uds</td>
                    <td>
                        ${isConfirmed
                            ? '<span class="badge bg-success"><i class="fas fa-check"></i> Lista</span>'
                            : '<span class="badge bg-warning text-dark"><i class="fas fa-hourglass-half"></i> Pendiente</span>'}
                         <button class="btn btn-sm btn-outline-danger border-0 ms-2 remove-br-box" data-box="${box.box_number}" title="Eliminar caja de reserva">
                            <i class="fas fa-trash-alt"></i>
                        </button>
                    </td>
                `;
                if (isConfirmed) tr.classList.add('table-success'); // Opcional: resaltar confirmadas
                tbody.appendChild(tr);
            });
            brBoxListContainer.appendChild(table);
        } else {
             brBoxListContainer.innerHTML = '<div class="text-center p-5 text-muted"><p>No hay cajas que coincidan con el filtro.</p></div>';
        }

        // Render Extras
        const extraContainer = document.getElementById('br_extra_list');
        extraContainer.innerHTML = '';
        document.getElementById('br_extra_count').textContent = brExtraBoxes.size;

        brExtraBoxes.forEach(boxNum => {
            const badge = document.createElement('div');
            badge.className = 'badge bg-warning text-dark me-2 mb-2 p-2';
            badge.innerHTML = `<i class="fas fa-plus"></i> ${boxNum}`;
            extraContainer.appendChild(badge);
        });

        // Completar button state
        const allConfirmed = confirmed >= total;
        brCompleteBtn.disabled = !allConfirmed;
        const statusSpan = document.getElementById('br_completion_status');
        if (allConfirmed) {
            statusSpan.textContent = "Reserva lista para completar";
            statusSpan.className = "text-success me-2 fw-bold";
        } else {
            statusSpan.textContent = `Faltan ${total - confirmed} cajas`;
            statusSpan.className = "text-muted me-2 small";
        }
    }

    async function handleBrScan(code) {
        if (!currentRes || !code) return;

        // Check if already is extra
        if (brExtraBoxes.has(code)) {
            showBrMessage('Esta caja ya está marcada como EXTRA', 'warning');
            brInput.value = '';
            return;
        }

        // Check if it's part of the reservation locally
        const boxItem = brBoxes.find(b => b.box_number === code);

        // Scenario: Box is IN reservation
        if (boxItem) {
            if (boxItem.status === 'confirmed') {
                showBrMessage('Esta caja YA estaba confirmada', 'info');
                brInput.value = '';
                return;
            }

            // Confirm it on backend
            showBrMessage('Confirmando...', 'info'); // Intent feedback
            try {
                const response = await fetch('/api/order-preparation/confirm-box-scan', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') },
                    body: JSON.stringify({ reservation_id: currentRes.id, box_number: code })
                });
                const data = await response.json();

                if (data.success) {
                    boxItem.status = 'confirmed';
                    showBrMessage('✅ Caja confirmada correctamente', 'success');
                    updateLastAction(`Confirmada: ${code}`);
                    renderBrState();
                } else {
                    showBrMessage(data.message, 'error');
                }
            } catch (e) { showBrMessage('Error de red', 'error'); }

            brInput.value = '';
            return;
        }

        // Scenario: Box is NOT in reservation (Ask to add as EXTRA)
        // Only ask if it's not already in extra (checked above)
        // Use a non-blocking UI ideally, but alert/confirm is standard for this legacy-style app
        const confirmExtra = confirm(`La caja ${code} NO está en esta reserva. ¿Deseas agregarla como EXTRA?`);
        if (confirmExtra) {
            addExtraBox(code);
        }
        brInput.value = '';
    }

    async function addExtraBox(code) {
        if (!code) return;

        try {
            const response = await fetch('/api/order-preparation/add-extra-box', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') },
                body: JSON.stringify({ reservation_id: currentRes.id, box_number: code })
            });
            const data = await response.json();

            if (data.success) {
                brExtraBoxes.add(code);
                showBrMessage('✅ Caja EXTRA añadida', 'success');
                updateLastAction(`Extra Añadida: ${code}`);
                renderBrState();
            } else {
                showBrMessage(data.message, 'error');
            }
        } catch (e) { showBrMessage('Error al añadir extra', 'error'); }

        brExtraInput.value = '';
    }

    function updateLastAction(text) {
        document.getElementById('br_last_action').style.display = 'block';
        document.getElementById('br_last_action_text').textContent = `${text} (${new Date().toLocaleTimeString()})`;
    }

    // Event Listeners for Tab 2
    brSelect.addEventListener('change', () => {
        brLoadBtn.disabled = !brSelect.value;
    });

    brLoadBtn.addEventListener('click', startReservation);

    brInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            const val = e.target.value.trim();
            if(val) handleBrScan(val);
        }
    });

    brAddExtraBtn.addEventListener('click', () => {
        const val = brExtraInput.value.trim();
        if(val) addExtraBox(val);
    });

    brExtraInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            const val = e.target.value.trim();
            if(val) addExtraBox(val);
        }
    });

    // Filters
    document.querySelectorAll('.btn-filter-br').forEach(btn => {
        btn.addEventListener('click', (e) => {
            document.querySelectorAll('.btn-filter-br').forEach(b => b.classList.remove('active'));
            e.target.classList.add('active');
            brFilter = e.target.dataset.filter;
            // Note: Filters might behave differently with the new split view (Pending vs Confirmed List)
            // But we keep renderBrState logic simple: Pending List always shows pending, Confirmed List shows confirmed.
            // The filter buttons might be redundant now or should only filter the confirmed list?
            // For now, I'll essentially ignore filter for Pending list as it should always be visible per request "below scanner"
            renderBrState();
        });
    });

    // Complete Button
    brCompleteBtn.addEventListener('click', async () => {
        if(!confirm('¿Estás seguro de completar esta reserva y generar el pedido? Esta acción no se puede deshacer.')) return;

        brCompleteBtn.disabled = true;
        brCompleteBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Procesando...';

        try {
            const response = await fetch('/api/order-preparation/complete-box-reservation', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') },
                body: JSON.stringify({ reservation_id: currentRes.id })
            });
            const data = await response.json();

            if (data.success) {
                alert('¡Reserva completada con éxito! Se ha generado el pedido.');
                location.reload(); // Reload to start fresh
            } else {
                showBrMessage(data.message, 'error');
                brCompleteBtn.disabled = false;
                brCompleteBtn.innerHTML = '<i class="fas fa-check-double"></i> Completar y Generar Pedido';
            }
        } catch (e) {
            alert('Error al completar reserva');
            brCompleteBtn.disabled = false;
        }
    });


    // Cancel Button
    brCancelBtn.addEventListener('click', async () => {
        if(!confirm('¿Estás SEGURO de cancelar esta reserva? \n\nESTA ACCIÓN CANCELARÁ LA RESERVA COMPLETAMENTE Y LIBERARÁ TODAS LAS CAJAS A DISPONIBLES.\n\nEsta acción no se puede deshacer.')) return;

        brCancelBtn.disabled = true;
        brCancelBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Cancelando...';

        try {
             const response = await fetch('/api/order-preparation/cancel-box-reservation', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') },
                body: JSON.stringify({ reservation_id: currentRes.id })
            });
            const data = await response.json();

            if (data.success) {
                alert('Reserva cancelada correctamente. Las cajas han sido liberadas.');
                location.reload();
            } else {
                showBrMessage(data.message, 'error');
                brCancelBtn.disabled = false;
                brCancelBtn.innerHTML = '<i class="fas fa-times-circle"></i> Cancelar Reserva';
            }
        } catch (e) {
            alert('Error al cancelar reserva');
            brCancelBtn.disabled = false;
            brCancelBtn.innerHTML = '<i class="fas fa-times-circle"></i> Cancelar Reserva';
        }
    });


    // Edit Details Logic
    const editModal = new bootstrap.Modal(document.getElementById('editReservationModal'));
    const editClientSelect = $('#edit_br_client');
    const saveDetailsBtn = document.getElementById('save_br_details_btn');

    document.getElementById('br_edit_details_btn').addEventListener('click', () => {
        if (!currentRes) return;

        // Populate form
        document.getElementById('edit_br_id').value = currentRes.id;
        document.getElementById('edit_br_notes').value = currentRes.notes || '';

        // Init select2 if needed or just set value
        // Re-use the clients loaded in tab 1 if compatible, or fetch again.
        // For simplicity, let's clone options from tab 1 client select
        editClientSelect.empty();
        $('#client_select option').each(function() {
            editClientSelect.append($(this).clone());
        });
        editClientSelect.val(currentRes.client_id).trigger('change');

        editClientSelect.select2({ dropdownParent: $('#editReservationModal') });

        editModal.show();
    });

    saveDetailsBtn.addEventListener('click', async () => {
        const newClientId = editClientSelect.val();
        const newNotes = document.getElementById('edit_br_notes').value;

        if (!newClientId) {
            alert('Debe seleccionar un cliente');
            return;
        }

        saveDetailsBtn.disabled = true;
        try {
            const response = await fetch('/api/order-preparation/update-reservation-details', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') },
                body: JSON.stringify({
                    reservation_id: currentRes.id,
                    client_id: newClientId,
                    notes: newNotes
                })
            });
            const data = await response.json();

            if (data.success) {
                // Update local model
                currentRes.client_id = newClientId;
                currentRes.notes = newNotes;
                // Update UI
                document.getElementById('br_client_name').textContent = data.client_name;
                document.getElementById('br_notes').textContent = data.notes || 'Sin notas';

                editModal.hide();
                showBrMessage('Detalles actualizados', 'success');
            } else {
                alert(data.message);
            }
        } catch (e) {
            alert('Error al guardar');
        } finally {
            saveDetailsBtn.disabled = false;
        }
    });

    // Remove Box Handler (Delegated)
    brBoxListContainer.addEventListener('click', async (e) => {
        const btn = e.target.closest('.remove-br-box');
        if (!btn) return;

        const boxNumber = btn.dataset.box;
        if (!confirm(`¿Estás seguro de ELIMINAR la caja ${boxNumber} de esta reserva? Pasará a estado Disponible.`)) return;

        try {
            const response = await fetch('/api/order-preparation/remove-box-reservation', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') },
                body: JSON.stringify({
                    reservation_id: currentRes.id,
                    box_number: boxNumber
                })
            });
            const data = await response.json();

            if (data.success) {
                // Remove from local array
                brBoxes = brBoxes.filter(b => b.box_number !== boxNumber);
                // Also remove from extra if present (though logic separates them usually)
                if (brExtraBoxes.has(boxNumber)) brExtraBoxes.delete(boxNumber);

                showBrMessage(`Caja ${boxNumber} eliminada`, 'success');
                renderBrState();
            } else {
                showBrMessage(data.message, 'error');
            }
        } catch (e) {
            showBrMessage('Error al eliminar caja', 'error');
        }
    });

    // Initial Load
    loadBoxReservations();


    // ==========================================
    // LOGICA TAB 3: PEDIDOS RESERVADOS (EXISTENTE)
    // ==========================================
    const reservedOrderSelect = document.getElementById('reserved_order_select');
    const loadReservedOrderBtn = document.getElementById('load_reserved_order_btn');
    const reservedOrderInfo = document.getElementById('reserved_order_info');
    const reservedBarcodeInput = document.getElementById('reserved_barcode_input');
    const extraBoxesInput = document.getElementById('extra_boxes_input');
    const confirmReservedOrderBtn = document.getElementById('confirm_reserved_order_btn');
    const cancelReservedOrderBtn = document.getElementById('cancel_reserved_order_btn');

    let currentReservedOrder = null;
    let reservedBoxes = [];
    let scannedReservedBoxes = new Set();
    let extraBoxes = new Set();

    // Cargar pedidos reservados
    async function loadReservedOrders() {
        try {
            const response = await fetch('/api/order-preparation/reserved-orders');
            const orders = await response.json();

            reservedOrderSelect.innerHTML = '<option value="">Seleccionar pedido reservado...</option>';
            orders.forEach(order => {
                const option = document.createElement('option');
                option.value = order.id;
                option.textContent = `Pedido #${order.id} - ${order.client?.client_name || 'Sin cliente'} (${order.order_date})`;
                reservedOrderSelect.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar pedidos reservados:', error);
            reservedOrderSelect.innerHTML = '<option value="">Error al cargar pedidos</option>';
        }
    }

    // Cargar un pedido reservado específico
    loadReservedOrderBtn.addEventListener('click', async () => {
        const orderId = reservedOrderSelect.value;
        if (!orderId) return;

        try {
            const response = await fetch(`/api/order-preparation/reserved-order/${orderId}`);
            const data = await response.json();

            if (data.success) {
                currentReservedOrder = data.order;
                reservedBoxes = data.boxes;
                scannedReservedBoxes.clear();
                extraBoxes.clear();

                // Mostrar información del pedido
                document.getElementById('reserved_order_id').textContent = `#${data.order.id}`;
                document.getElementById('reserved_order_client').textContent = data.order.client?.client_name || 'Sin cliente';
                document.getElementById('reserved_boxes_count').textContent = data.total_boxes;
                document.getElementById('reserved_items_count').textContent = data.total_items;

                // Mostrar cajas reservadas
                const reservedBoxesList = document.getElementById('reserved_boxes_list');
                reservedBoxesList.innerHTML = '';
                reservedBoxes.forEach(box => {
                    const div = document.createElement('div');
                    div.className = 'badge bg-secondary me-2 mb-2';
                    div.textContent = box;
                    div.id = `reserved-box-${box}`;
                    reservedBoxesList.appendChild(div);
                });

                // Limpiar listas de escaneadas y extra
                document.getElementById('scanned_boxes_list').innerHTML = '<p class="text-muted text-center mb-0">Aún no se han escaneado cajas</p>';
                document.getElementById('extra_boxes_list').style.display = 'none';

                reservedOrderInfo.style.display = 'block';
                reservedBarcodeInput.focus();
                updateConfirmReservedButton();
            }
        } catch (error) {
            console.error('Error al cargar pedido reservado:', error);
            showReservedMessage('Error al cargar el pedido reservado', 'error');
        }
    });

    reservedOrderSelect.addEventListener('change', () => {
        loadReservedOrderBtn.disabled = !reservedOrderSelect.value;
    });

    // Escanear cajas para confirmar pedido reservado
    reservedBarcodeInput.addEventListener('keypress', async (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            const boxNumber = reservedBarcodeInput.value.trim();
            if (!boxNumber) return;

            // Validar que la caja esté en el pedido reservado
            if (!reservedBoxes.includes(boxNumber)) {
                showReservedMessage(`La caja ${boxNumber} no está en este pedido reservado`, 'error');
                reservedBarcodeInput.value = '';
                return;
            }

            // Validar que no esté ya escaneada
            if (scannedReservedBoxes.has(boxNumber)) {
                showReservedMessage(`La caja ${boxNumber} ya fue escaneada`, 'error');
                reservedBarcodeInput.value = '';
                return;
            }

            // Validar físicamente con el servidor
            try {
                const response = await fetch('/api/order-preparation/validate-reserved-box', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({
                        order_id: currentReservedOrder.id,
                        box_number: boxNumber
                    })
                });

                const data = await response.json();

                if (data.is_in_order) {
                    scannedReservedBoxes.add(boxNumber);
                    updateScannedBoxesList();
                    updateReservedBoxesDisplay();
                    updateConfirmReservedButton();
                    showReservedMessage(`Caja ${boxNumber} confirmada correctamente`, 'success');
                    reservedBarcodeInput.value = '';
                } else {
                    showReservedMessage(`Error: ${data.message}`, 'error');
                    reservedBarcodeInput.value = '';
                }
            } catch (error) {
                console.error('Error al validar caja:', error);
                showReservedMessage('Error de comunicación al validar la caja', 'error');
            }
        }
    });

    // Escanear cajas extra
    extraBoxesInput.addEventListener('keypress', async (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            const boxNumber = extraBoxesInput.value.trim();
            if (!boxNumber) return;

            // Validar que no esté ya en las reservadas o escaneadas
            if (reservedBoxes.includes(boxNumber) || scannedReservedBoxes.has(boxNumber)) {
                showReservedMessage(`La caja ${boxNumber} ya está en el pedido`, 'error');
                extraBoxesInput.value = '';
                return;
            }

            if (extraBoxes.has(boxNumber)) {
                showReservedMessage(`La caja ${boxNumber} ya fue añadida como extra`, 'error');
                extraBoxesInput.value = '';
                return;
            }

            // Validar que la caja esté disponible
            try {
                const response = await fetch('/api/order-preparation/validate-box', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({ input: boxNumber })
                });

                const data = await response.json();

                if (data.exists) {
                    extraBoxes.add(boxNumber);
                    updateExtraBoxesList();
                    showReservedMessage(`Caja extra ${boxNumber} añadida`, 'success');
                    extraBoxesInput.value = '';
                } else {
                    showReservedMessage(`La caja ${boxNumber} no está disponible`, 'error');
                    extraBoxesInput.value = '';
                }
            } catch (error) {
                console.error('Error al validar caja extra:', error);
                showReservedMessage('Error de comunicación al validar la caja', 'error');
            }
        }
    });

    // Actualizar lista de cajas escaneadas
    function updateScannedBoxesList() {
        const scannedList = document.getElementById('scanned_boxes_list');
        if (scannedReservedBoxes.size === 0) {
            scannedList.innerHTML = '<p class="text-muted text-center mb-0">Aún no se han escaneado cajas</p>';
            return;
        }

        scannedList.innerHTML = '';
        Array.from(scannedReservedBoxes).forEach(box => {
            const div = document.createElement('div');
            div.className = 'badge bg-success me-2 mb-2';
            div.textContent = box;
            scannedList.appendChild(div);
        });
    }

    // Actualizar visualización de cajas reservadas (marcar las escaneadas)
    function updateReservedBoxesDisplay() {
        reservedBoxes.forEach(box => {
            const element = document.getElementById(`reserved-box-${box}`);
            if (element) {
                if (scannedReservedBoxes.has(box)) {
                    element.className = 'badge bg-success me-2 mb-2';
                } else {
                    element.className = 'badge bg-secondary me-2 mb-2';
                }
            }
        });
    }

    // Actualizar lista de cajas extra
    function updateExtraBoxesList() {
        const extraList = document.getElementById('extra_boxes_list').querySelector('.border');
        if (extraBoxes.size === 0) {
            document.getElementById('extra_boxes_list').style.display = 'none';
            return;
        }

        document.getElementById('extra_boxes_list').style.display = 'block';
        extraList.innerHTML = '';
        Array.from(extraBoxes).forEach(box => {
            const div = document.createElement('div');
            div.className = 'badge bg-warning text-dark me-2 mb-2';
            div.textContent = box;
            const removeBtn = document.createElement('span');
            removeBtn.className = 'ms-2 cursor-pointer';
            removeBtn.innerHTML = '×';
            removeBtn.style.cursor = 'pointer';
            removeBtn.onclick = () => {
                extraBoxes.delete(box);
                updateExtraBoxesList();
            };
            div.appendChild(removeBtn);
            extraList.appendChild(div);
        });
    }

    // Actualizar estado del botón de confirmar
    function updateConfirmReservedButton() {
        const allScanned = reservedBoxes.every(box => scannedReservedBoxes.has(box));
        confirmReservedOrderBtn.disabled = !allScanned || scannedReservedBoxes.size === 0;
    }

    // Mostrar mensaje
    function showReservedMessage(message, type) {
        const messageDiv = document.getElementById('reserved_order_message');
        messageDiv.textContent = message;
        messageDiv.className = `message ${type}`;
        messageDiv.style.display = 'block';
        setTimeout(() => {
            messageDiv.style.display = 'none';
        }, 3000);
    }

    // Confirmar pedido reservado
    confirmReservedOrderBtn.addEventListener('click', async () => {
        if (!currentReservedOrder) return;

        const allScanned = reservedBoxes.every(box => scannedReservedBoxes.has(box));
        if (!allScanned) {
            showReservedMessage('Debes escanear todas las cajas reservadas antes de confirmar', 'error');
            return;
        }

        if (!confirm('¿Confirmar y despachar este pedido reservado? Esta acción no se puede deshacer.')) {
            return;
        }

        confirmReservedOrderBtn.disabled = true;
        confirmReservedOrderBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Confirmando...';

        try {
            const response = await fetch('/api/order-preparation/confirm-reserved-order', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    order_id: currentReservedOrder.id,
                    scanned_boxes: Array.from(scannedReservedBoxes),
                    extra_boxes: Array.from(extraBoxes)
                })
            });

            const data = await response.json();

            if (data.success) {
                showReservedMessage(data.message, 'success');
                setTimeout(() => {
                    location.reload();
                }, 2000);
            } else {
                showReservedMessage(data.message || 'Error al confirmar el pedido', 'error');
                confirmReservedOrderBtn.disabled = false;
                confirmReservedOrderBtn.innerHTML = '<i class="fas fa-check"></i> Confirmar y Despachar Pedido';
            }
        } catch (error) {
            console.error('Error al confirmar pedido:', error);
            showReservedMessage('Error de comunicación al confirmar el pedido', 'error');
            confirmReservedOrderBtn.disabled = false;
            confirmReservedOrderBtn.innerHTML = '<i class="fas fa-check"></i> Confirmar y Despachar Pedido';
        }
    });

    // Cancelar confirmación de pedido reservado
    cancelReservedOrderBtn.addEventListener('click', () => {
        reservedOrderInfo.style.display = 'none';
        currentReservedOrder = null;
        reservedBoxes = [];
        scannedReservedBoxes.clear();
        extraBoxes.clear();
        reservedOrderSelect.value = '';
        loadReservedOrderBtn.disabled = true;
    });

    // Cargar pedidos reservados al iniciar
    loadReservedOrders();
});

</script>
@endpush
