@extends('layouts.app')

@section('title', __('inventory.title'))

@section('content')
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-boxes"></i> {{ __('inventory.title') }}</h2>
        <div class="d-flex gap-2">
            @can('inventory.view')
                <a href="#" id="export-excel-btn" class="btn btn-secondary">
                    <i class="fas fa-file-excel"></i> {{ __('inventory.buttons.export_excel') }}
                </a>
                <a href="#" id="export-grouped-btn" class="btn btn-info">
                    <i class="fas fa-file-excel"></i> {{ __('inventory.buttons.export_grouped') }}
                </a>
                <button type="button" id="export-brand-btn" class="btn btn-warning">
                    <i class="fas fa-file-excel"></i> Exportar por Marca
                </button>
            @endcan
            @can('inventory.create')
                <a href="{{ route('inventory.create') }}" class="btn">
                    <i class="fas fa-plus"></i> {{ __('inventory.buttons.add_item') }}
                </a>
            @endcan
        </div>
    </div>

    <p>{{ __('inventory.description') }}</p>

    <!-- Filtros -->
    <div class="card mb-3">
        <div class="card-header">
            <h5><i class="fas fa-filter"></i> {{ __('inventory.filters.title') }}</h5>
        </div>
        <div class="card-body">
            <form id="search-form" class="row g-3">
                <div class="col-md-2">
                    <label for="search_barcode" class="form-label">{{ __('inventory.filters.barcode') }}</label>
                    <input type="text" class="form-control" id="search_barcode"
                        placeholder="{{ __('inventory.filters.barcode_placeholder') }}">
                </div>
                <div class="col-md-2">
                    <label for="search_mocaco" class="form-label">{{ __('inventory.filters.mocaco') }}</label>
                    <input type="text" class="form-control" id="search_mocaco"
                        placeholder="{{ __('inventory.filters.mocaco_placeholder') }}">
                </div>
                <div class="col-md-2">
                    <label for="search_n_carton" class="form-label">{{ __('inventory.filters.box_number') }}</label>
                    <input type="text" class="form-control" id="search_n_carton"
                        placeholder="{{ __('inventory.filters.box_number_placeholder') }}">
                </div>
                <div class="col-md-2">
                    <label for="search_ubicacion" class="form-label">{{ __('inventory.filters.location') }}</label>
                    <input type="text" class="form-control" id="search_ubicacion"
                        placeholder="{{ __('inventory.filters.location_placeholder') }}">
                </div>
                <div class="col-md-2">
                    <label for="search_containers" class="form-label">Container</label>
                    <div class="multiselect-wrapper">
                        <button type="button" class="form-control multiselect-button" id="search_containers_btn">
                            <span class="multiselect-text">Seleccionar...</span>
                            <i class="fas fa-chevron-down"></i>
                        </button>
                        <div class="multiselect-dropdown" id="search_containers_dropdown" style="display: none;">
                            <div class="multiselect-header">
                                <button type="button" class="btn btn-sm btn-link" id="containers_select_all">Seleccionar Todos</button>
                                <button type="button" class="btn btn-sm btn-link" id="containers_clear">Limpiar</button>
                            </div>
                            <div class="multiselect-search">
                                <input type="text" class="form-control form-control-sm" id="containers_search_input" placeholder="Buscar container...">
                            </div>
                            <div class="multiselect-options" id="search_containers_options">
                                <div class="text-muted text-center p-2">Cargando...</div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-2">
                    <label for="search_seasons" class="form-label">Internal Season</label>
                    <div class="multiselect-wrapper">
                        <button type="button" class="form-control multiselect-button" id="search_seasons_btn">
                            <span class="multiselect-text">Seleccionar...</span>
                            <i class="fas fa-chevron-down"></i>
                        </button>
                        <div class="multiselect-dropdown" id="search_seasons_dropdown" style="display: none;">
                            <div class="multiselect-header">
                                <button type="button" class="btn btn-sm btn-link" id="seasons_select_all">Seleccionar Todos</button>
                                <button type="button" class="btn btn-sm btn-link" id="seasons_clear">Limpiar</button>
                            </div>
                            <div class="multiselect-search">
                                <input type="text" class="form-control form-control-sm" id="seasons_search_input" placeholder="Buscar temporada...">
                            </div>
                            <div class="multiselect-options" id="search_seasons_options">
                                <div class="text-muted text-center p-2">Cargando...</div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-2">
                    <label for="search_marcas" class="form-label">{{ __('inventory.filters.brand') }}</label>
                    <input type="text" class="form-control" id="search_marcas"
                        placeholder="{{ __('inventory.filters.brand_placeholder') }}">
                </div>
                <div class="col-md-2">
                    <div class="form-check mt-4">
                        <input class="form-check-input" type="checkbox" id="search_needs_review" value="1">
                        <label class="form-check-label" for="search_needs_review">
                            <i class="fas fa-exclamation-circle text-warning"></i> Solo "Sin Info"
                        </label>
                    </div>
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="fas fa-search"></i> {{ __('inventory.filters.search_button') }}
                    </button>
                </div>
            </form>
        </div>
    </div>

    <div id="search_message" class="alert" style="display:none;"></div>

    <!-- Controles de Paginación -->
    <div class="card mb-3">
        <div class="card-body">
            <div class="d-flex justify-content-between align-items-center flex-wrap gap-3">
                <div class="d-flex align-items-center gap-2">
                    <label for="items_per_page">{{ __('inventory.pagination.rows_per_page') }}</label>
                    <select id="items_per_page" class="form-control" style="width: auto;">
                        <option value="10">10</option>
                        <option value="25">25</option>
                        <option value="50" selected>50</option>
                        <option value="100">100</option>
                        <option value="250">250</option>
                    </select>
                </div>
                <div id="pagination_info" class="text-muted"></div>
                <div class="d-flex align-items-center gap-2">
                    <button id="prev_page_btn" class="btn btn-sm"
                        disabled>{{ __('inventory.pagination.previous') }}</button>
                    <span id="page_numbers"></span>
                    <button id="next_page_btn" class="btn btn-sm" disabled>{{ __('inventory.pagination.next') }}</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Tabla de Resultados -->
    <div class="card">
        <div class="card-header">
            <h5><i class="fas fa-list"></i> {{ __('inventory.results.title') }}</h5>
        </div>
        <div class="card-body p-0">
            <div id="inventory_search_results">
                <p class="text-center text-muted p-4">{{ __('inventory.results.initial_message') }}</p>
            </div>
        </div>
    </div>

    <!-- Modal de Confirmación de Eliminación -->
    <div id="deleteModal" class="modal-overlay" style="display: none;">
        <div class="modal-content">
            <h3>{{ __('inventory.modal.delete_title') }}</h3>
            {{-- <p>Ingrese el código secreto para confirmar la eliminación:</p> --}}
            {{-- <input type="text" id="secret_code_input" class="form-control mb-3" placeholder="Código secreto"> --}}
            <div class="d-flex justify-content-end gap-2">
                <button id="cancel_delete" class="btn btn-secondary">{{ __('inventory.modal.cancel') }}</button>
                <button id="confirm_delete" class="btn btn-danger">{{ __('inventory.modal.confirm') }}</button>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <style>
        .pagination-link {
            padding: 5px 10px;
            margin: 0 2px;
            border: 1px solid #007bff;
            border-radius: 3px;
            text-decoration: none;
            color: #007bff;
            cursor: pointer;
            font-size: 14px;
            display: inline-block;
        }

        .pagination-link:hover:not(.active) {
            background-color: #e6f2ff;
        }

        .pagination-link.active {
            background-color: #007bff;
            color: white;
            pointer-events: none;
        }

        .sortable {
            cursor: pointer;
            position: relative;
            padding-right: 25px;
        }

        .sortable::after {
            content: '\2195';
            position: absolute;
            right: 8px;
            top: 50%;
            transform: translateY(-50%);
            font-size: 0.9em;
            color: #888;
        }

        .sortable.asc::after {
            content: '\2191';
            color: #007bff;
        }

        .sortable.desc::after {
            content: '\2193';
            color: #007bff;
        }

        .editable {
            background-color: #fffacd;
            cursor: text;
        }

        .editable:hover {
            background-color: #ffe082;
        }

        .editable:focus {
            outline: 2px solid #007bff;
            background-color: #ffffff;
        }

        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            z-index: 1000;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background: #fff;
            padding: 25px;
            border-radius: 8px;
            max-width: 400px;
            width: 90%;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.3);
        }

        .edit-btn {
            background-color: #ffc107;
            color: #000;
            border: none;
            border-radius: 4px;
            padding: 3px 8px;
            cursor: pointer;
            font-size: 0.75em;
            margin-right: 5px;
        }

        .edit-btn:hover {
            background-color: #e0a800;
        }

        .delete-btn {
            background-color: #dc3545;
            color: white;
            border: none;
            border-radius: 4px;
            padding: 3px 8px;
            cursor: pointer;
            font-size: 0.75em;
        }

        .delete-btn:hover {
            background-color: #c82333;
        }

        .status-reservado {
            background-color: #fff3cd;
            color: #856404;
            font-weight: bold;
            padding: 5px 8px;
            border-radius: 4px;
        }

        .status-disponible {
            background-color: #d4edda;
            color: #155724;
            padding: 3px 6px;
            border-radius: 4px;
        }

        .status-despachado {
            background-color: #d1ecf1;
            color: #0c5460;
            padding: 3px 6px;
            border-radius: 4px;
        }

        .status-cell {
            white-space: nowrap;
            max-width: 300px;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        /* Multiselect Styles */
        .multiselect-wrapper {
            position: relative;
        }

        .multiselect-button {
            display: flex;
            justify-content: space-between;
            align-items: center;
            text-align: left;
            background-color: white;
            cursor: pointer;
        }

        .multiselect-button:hover {
            background-color: #f8f9fa;
        }

        .multiselect-dropdown {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid #ced4da;
            border-radius: 0.25rem;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
            z-index: 1000;
            margin-top: 2px;
            max-height: 300px;
            overflow-y: auto;
        }

        .multiselect-header {
            display: flex;
            justify-content: space-between;
            padding: 8px 12px;
            border-bottom: 1px solid #dee2e6;
            background-color: #f8f9fa;
        }

        .multiselect-header .btn-link {
            padding: 0;
            font-size: 0.875rem;
            text-decoration: none;
        }

        .multiselect-search {
            padding: 8px 12px;
            border-bottom: 1px solid #dee2e6;
            background-color: #fff;
        }

        .multiselect-search input {
            width: 100%;
            border: 1px solid #ced4da;
            border-radius: 0.25rem;
            padding: 4px 8px;
        }

        .multiselect-search input:focus {
            outline: none;
            border-color: #80bdff;
            box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
        }

        .multiselect-options {
            padding: 8px 0;
        }

        .multiselect-option {
            padding: 6px 12px;
            cursor: pointer;
            display: flex;
            align-items: center;
        }

        .multiselect-option:hover {
            background-color: #f8f9fa;
        }

        .multiselect-option input[type="checkbox"] {
            margin-right: 8px;
        }

        .multiselect-count {
            background-color: #007bff;
            color: white;
            border-radius: 10px;
            padding: 2px 8px;
            font-size: 0.75rem;
            margin-left: 8px;
        }
    </style>
@endpush

@php
    $displayColumnsConfig = [
        ['id' => 'cadena', 'header' => __('inventory.table.headers.brand'), 'sortable' => true],
        ['id' => 'campana', 'header' => __('inventory.table.headers.season'), 'sortable' => true],
        [
            'id' => 'ubicacion',
            'header' => __('inventory.table.headers.location'),
            'sortable' => true,
            'editable' => true,
        ],
        ['id' => 'n_carton', 'header' => __('inventory.table.headers.box'), 'sortable' => true],
        ['id' => 'seccion', 'header' => __('inventory.table.headers.section'), 'sortable' => true],
        [
            'id' => 'familia_articulo_description',
            'header' => __('inventory.table.headers.description'),
            'sortable' => true,
        ],
        [
            'id' => 'famillie_usuario',
            'header' => __('inventory.table.headers.family'),
            'sortable' => true,
            'editable' => true,
        ],
        [
            'id' => 'detail_usuario',
            'header' => __('inventory.table.headers.detail'),
            'sortable' => true,
            'editable' => true,
        ],
        ['id' => 'full_barcode', 'header' => __('inventory.table.headers.barcode'), 'sortable' => true],
        ['id' => 'mocaco', 'header' => __('inventory.table.headers.mocaco'), 'sortable' => true],
        [
            'id' => 'season_int',
            'header' => __('inventory.table.headers.season_it'),
            'sortable' => true,
            'editable' => true,
        ],
        ['id' => 'notes', 'header' => __('inventory.table.headers.notes'), 'sortable' => true, 'editable' => true],
        [
            'id' => 'categoria_seleccionada',
            'header' => __('inventory.table.headers.category'),
            'sortable' => true,
            'editable' => true,
        ],
        ['id' => 'conteneur', 'header' => __('inventory.table.headers.container'), 'sortable' => true],
        ['id' => 'status_display', 'header' => __('inventory.table.headers.status'), 'sortable' => true],
        ['id' => 'fecha_escaneo', 'header' => 'Fecha Escaneo', 'sortable' => true],
        ['id' => 'cliente', 'header' => __('inventory.table.headers.client'), 'sortable' => true],
        ['id' => 'pedido_id', 'header' => __('inventory.table.headers.order_id'), 'sortable' => true],
        ['id' => 'actions', 'header' => __('inventory.table.headers.actions'), 'sortable' => false],
    ];

    $inventoryTexts = [
        'loading' => __('inventory.messages.loading'),
        'initial' => __('inventory.results.initial_message'),
        'no_results' => __('inventory.messages.no_results'),
        'search_error' => __('inventory.messages.search_error'),
        'server_error' => __('inventory.messages.server_error'),
        'found' => __('inventory.messages.found', ['count' => ':count', 'total' => ':total']),
        'update_success' => __('inventory.messages.update_success'),
        'update_error' => __('inventory.messages.update_error', ['message' => ':message']),
        'delete_success' => __('inventory.messages.delete_success'),
        'delete_error' => __('inventory.messages.delete_error', ['message' => ':message']),
        'pagination_info' => __('inventory.pagination.info', [
            'current' => ':current',
            'total' => ':total',
            'records' => ':records',
        ]),
        'results_empty' => __('inventory.results.empty_message'),
    ];
@endphp

@push('scripts')
    <script>
        // Permisos del usuario para verificar en JavaScript
        const userPermissions = {
            canUpdate: @json(auth()->user()->can('inventory.update')),
            canDelete: @json(auth()->user()->can('inventory.delete')),
        };

        const displayColumns = @json($displayColumnsConfig);
        const inventoryTexts = @json($inventoryTexts);

        document.addEventListener('DOMContentLoaded', function() {
            const searchBarcode = document.getElementById('search_barcode');
            const searchMocaco = document.getElementById('search_mocaco');
            const searchNCarton = document.getElementById('search_n_carton');
            const searchUbicacion = document.getElementById('search_ubicacion');
            const searchMarcas = document.getElementById('search_marcas');
            const searchForm = document.getElementById('search-form');
            const searchMessageDiv = document.getElementById('search_message');
            const inventorySearchResults = document.getElementById('inventory_search_results');
            const itemsPerPageSelect = document.getElementById('items_per_page');
            const prevPageBtn = document.getElementById('prev_page_btn');
            const nextPageBtn = document.getElementById('next_page_btn');
            const pageNumbersSpan = document.getElementById('page_numbers');
            const paginationInfoDiv = document.getElementById('pagination_info');

            let currentPage = 1;
            let itemsPerPage = parseInt(itemsPerPageSelect.value);
            let totalRecords = 0;
            let currentSortColumn = 'fecha_escaneo';
            let currentSortDirection = 'desc';
            let lastFetchedResults = [];
            let itemToDelete = null;

            // Multiselect state
            let selectedContainers = [];
            let selectedSeasons = [];
            let filterOptions = { containers: [], seasons: [], brands: [] };

            const editableColumns = displayColumns.filter(col => col.editable).map(col => col.id);

            function formatMessage(template, replacements = {}) {
                return template.replace(/:([a-zA-Z_]+)/g, (match, key) => {
                    return Object.prototype.hasOwnProperty.call(replacements, key) ? replacements[key] :
                        match;
                });
            }

            function showSearchMessage(msg, type) {
                searchMessageDiv.textContent = msg;
                searchMessageDiv.className = `alert alert-${type}`;
                searchMessageDiv.style.display = 'block';
                setTimeout(() => {
                    searchMessageDiv.style.display = 'none';
                }, 3000);
            }

            function updatePaginationControls() {
                const totalPages = Math.ceil(totalRecords / itemsPerPage);
                prevPageBtn.disabled = currentPage === 1;
                nextPageBtn.disabled = currentPage === totalPages || totalPages === 0;

                pageNumbersSpan.innerHTML = '';
                if (totalPages > 0) {
                    let startPage = Math.max(1, currentPage - 2);
                    let endPage = Math.min(totalPages, currentPage + 2);

                    if (currentPage <= 3 && totalPages > 5) {
                        endPage = 5;
                    } else if (currentPage >= totalPages - 2 && totalPages > 5) {
                        startPage = totalPages - 4;
                    }

                    for (let i = startPage; i <= endPage; i++) {
                        const pageLink = document.createElement('a');
                        pageLink.href = '#';
                        pageLink.textContent = i;
                        pageLink.classList.add('pagination-link');
                        if (i === currentPage) {
                            pageLink.classList.add('active');
                        }
                        pageLink.addEventListener('click', (e) => {
                            e.preventDefault();
                            if (currentPage !== i) {
                                currentPage = i;
                                performSearch();
                            }
                        });
                        pageNumbersSpan.appendChild(pageLink);
                    }
                }
                paginationInfoDiv.textContent =
                    `Página ${totalPages === 0 ? 0 : currentPage} de ${totalPages} (${totalRecords} registros)`;
            }

            async function performSearch() {
                inventorySearchResults.innerHTML =
                    `<p class="text-center text-muted p-4">${inventoryTexts.loading}</p>`;

                const searchParams = {
                    barcode: searchBarcode.value.trim(),
                    mocaco: searchMocaco.value.trim(),
                    n_carton: searchNCarton.value.trim(),
                    ubicacion: searchUbicacion.value.trim(),
                    marcas: searchMarcas.value.trim(),
                    containers: selectedContainers.length > 0 ? selectedContainers : undefined,
                    seasons: selectedSeasons.length > 0 ? selectedSeasons : undefined,
                    needs_review: document.getElementById('search_needs_review').checked ? true : undefined,
                    page: currentPage,
                    limit: itemsPerPage
                };

                try {
                    const response = await fetch('/inventory/search', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'Accept': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')
                                .getAttribute('content')
                        },
                        body: JSON.stringify(searchParams)
                    });

                    const data = await response.json();

                    if (data.success) {
                        totalRecords = data.totalRecords || 0;
                        updatePaginationControls();

                        if (data.results.length > 0) {
                            lastFetchedResults = data.results;

                            if (currentSortColumn) {
                                sortResults(currentSortColumn, currentSortDirection);
                            }

                            renderTable(lastFetchedResults);
                            showSearchMessage(formatMessage(inventoryTexts.found, {
                                count: data.results.length,
                                total: totalRecords
                            }), 'success');
                        } else {
                            inventorySearchResults.innerHTML =
                                `<p class="text-center text-muted p-4">${inventoryTexts.results_empty}</p>`;
                            showSearchMessage(inventoryTexts.no_results, 'danger');
                            lastFetchedResults = [];
                        }
                    } else {
                        inventorySearchResults.innerHTML =
                            `<p class="text-center text-muted p-4">${inventoryTexts.search_error}</p>`;
                        showSearchMessage(data.message || inventoryTexts.search_error, 'danger');
                        lastFetchedResults = [];
                    }
                } catch (error) {
                    console.error('Error de comunicación:', error);
                    showSearchMessage(inventoryTexts.server_error, 'danger');
                    lastFetchedResults = [];
                }
            }

            function renderTable(results) {
                let tableHtml = '<div class="table-responsive"><table class="data-table"><thead><tr>';
                displayColumns.forEach(col => {
                    const isSortable = col.sortable ? 'sortable' : '';
                    let sortClass = '';
                    if (col.id === currentSortColumn) {
                        sortClass = currentSortDirection;
                    }
                    tableHtml +=
                        `<th data-column-id="${col.id}" class="${isSortable} ${sortClass}">${col.header}</th>`;
                });
                tableHtml += '</tr></thead><tbody>';

                results.forEach(itemData => {
                    tableHtml += `<tr data-id="${itemData.id}">`;
                    displayColumns.forEach(col => {
                        let cellValue = itemData[col.id];
                        // Solo hacer editable si el usuario tiene permiso de actualizar
                        let isEditable = (editableColumns.includes(col.id) && userPermissions
                            .canUpdate) ? 'contenteditable="true" class="editable"' : '';
                        let dataCol = col.id;

                        if (col.id === 'actions') {
                            let actionsHtml = '';
                            if (userPermissions.canUpdate) {
                                actionsHtml +=
                                    `<button class="edit-btn" data-id="${itemData.id}" title="Editar">✏️</button> `;
                            }
                            if (userPermissions.canDelete) {
                                actionsHtml +=
                                    `<button class="delete-btn" data-id="${itemData.id}">X</button>`;
                            }
                            tableHtml += `<td>${actionsHtml || '-'}</td>`;
                        } else if (col.id === 'status_display') {
                            // Status con estilos especiales
                            if (cellValue === undefined || cellValue === null || cellValue === '') {
                                cellValue = '';
                            }
                            let statusClass = 'status-cell';
                            if (cellValue && cellValue.toLowerCase().includes('reservado')) {
                                statusClass += ' status-reservado';
                            } else if (cellValue && cellValue.toLowerCase().includes(
                                    'disponible')) {
                                statusClass += ' status-disponible';
                            } else if (cellValue && cellValue.toLowerCase().includes(
                                    'despachado')) {
                                statusClass += ' status-despachado';
                            } else if (cellValue && (cellValue.toLowerCase() === 'order_proposal' || cellValue === 'Propuesta de pedido')) {
                                cellValue = 'Propuesta de pedido';
                                statusClass += ' status-reservado'; // Reusing reserved style or create a new one
                            }
                            tableHtml +=
                                `<td data-col="${dataCol}" class="${statusClass}" title="${cellValue}">${cellValue}</td>`;
                        } else if (col.id === 'pedido_id') {
                            // ID Pedido como enlace si tiene valor
                            if (cellValue === undefined || cellValue === null || cellValue === '') {
                                tableHtml += `<td data-col="${dataCol}">-</td>`;
                            } else {
                                tableHtml +=
                                    `<td data-col="${dataCol}"><a href="/orders/${cellValue}" target="_blank" style="color: #007bff; text-decoration: none;">#${cellValue}</a></td>`;
                            }
                        } else {
                            if (cellValue === undefined || cellValue === null || cellValue === '') {
                                cellValue = '';
                            }
                            tableHtml +=
                                `<td data-col="${dataCol}" ${isEditable}>${cellValue}</td>`;
                        }
                    });
                    tableHtml += '</tr>';
                });

                tableHtml += '</tbody></table></div>';
                inventorySearchResults.innerHTML = tableHtml;

                attachEditListeners();
                attachSortListeners();
                attachDeleteListeners();
                attachEditButtonListeners();
            }

            function attachEditListeners() {
                const editableCells = inventorySearchResults.querySelectorAll('td[contenteditable="true"]');
                editableCells.forEach(cell => {
                    cell.removeEventListener('blur', handleCellEdit);
                    cell.removeEventListener('keydown', handleCellKeydown);
                    cell.addEventListener('blur', handleCellEdit);
                    cell.addEventListener('keydown', handleCellKeydown);
                });
            }

            function handleCellKeydown(event) {
                if (event.key === 'Enter') {
                    event.preventDefault();
                    event.target.blur();
                }
            }

            async function handleCellEdit(event) {
                const cell = event.target;
                const newValue = cell.textContent.trim();
                const itemId = cell.closest('tr').dataset.id;
                const columnToUpdate = cell.dataset.col;

                if (!itemId || !columnToUpdate) {
                    console.error("No se pudo identificar la fila o columna para actualizar.");
                    return;
                }

                const originalItem = lastFetchedResults.find(item => item.id == itemId);
                const originalValue = originalItem ? originalItem[columnToUpdate] : null;

                if (newValue === String(originalValue || '')) {
                    return;
                }

                try {
                    const response = await fetch('/inventory/update-field', {
                        method: 'PUT',
                        headers: {
                            'Content-Type': 'application/json',
                            'Accept': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')
                                .getAttribute('content')
                        },
                        body: JSON.stringify({
                            id: itemId,
                            column: columnToUpdate,
                            value: newValue
                        })
                    });

                    const data = await response.json();

                    if (data.success) {
                        showSearchMessage(inventoryTexts.update_success, 'success');
                        const updatedItemIndex = lastFetchedResults.findIndex(item => item.id == itemId);
                        if (updatedItemIndex !== -1) {
                            lastFetchedResults[updatedItemIndex][columnToUpdate] = newValue;
                        }
                    } else {
                        showSearchMessage(formatMessage(inventoryTexts.update_error, {
                            message: data.message || ''
                        }), 'danger');
                        if (originalItem) {
                            cell.textContent = originalValue;
                        }
                    }
                } catch (error) {
                    console.error('Error de comunicación al actualizar:', error);
                    showSearchMessage(inventoryTexts.server_error, 'danger');
                    if (originalItem) {
                        cell.textContent = originalValue;
                    }
                }
            }

            function attachSortListeners() {
                const sortableHeaders = inventorySearchResults.querySelectorAll('th.sortable');
                sortableHeaders.forEach(header => {
                    header.removeEventListener('click', handleHeaderClick);
                    header.addEventListener('click', handleHeaderClick);
                });
            }

            function handleHeaderClick(event) {
                const columnId = event.currentTarget.dataset.columnId;
                if (currentSortColumn === columnId) {
                    currentSortDirection = currentSortDirection === 'asc' ? 'desc' : 'asc';
                } else {
                    currentSortColumn = columnId;
                    currentSortDirection = 'asc';
                }
                sortResults(currentSortColumn, currentSortDirection);
                renderTable(lastFetchedResults);
            }

            function sortResults(column, direction) {
                lastFetchedResults.sort((a, b) => {
                    let valA = a[column];
                    let valB = b[column];

                    if (valA === null || valA === undefined || valA === '') {
                        valA = (direction === 'asc') ? Infinity : -Infinity;
                    }
                    if (valB === null || valB === undefined || valB === '') {
                        valB = (direction === 'asc') ? Infinity : -Infinity;
                    }

                    const numA = parseFloat(valA);
                    const numB = parseFloat(valB);

                    if (!isNaN(numA) && !isNaN(numB) && isFinite(numA) && isFinite(numB)) {
                        return direction === 'asc' ? numA - numB : numB - numA;
                    } else {
                        const strA = String(valA).toLowerCase();
                        const strB = String(valB).toLowerCase();
                        if (strA < strB) return direction === 'asc' ? -1 : 1;
                        if (strA > strB) return direction === 'asc' ? 1 : -1;
                        return 0;
                    }
                });

                document.querySelectorAll('th.sortable').forEach(th => {
                    th.classList.remove('asc', 'desc');
                    if (th.dataset.columnId === column) {
                        th.classList.add(direction);
                    }
                });
            }

            function attachEditButtonListeners() {
                const editButtons = inventorySearchResults.querySelectorAll('.edit-btn');
                editButtons.forEach(button => {
                    button.removeEventListener('click', handleEditButtonClick);
                    button.addEventListener('click', handleEditButtonClick);
                });
            }

            function handleEditButtonClick(event) {
                const recordId = event.target.dataset.id;
                if (recordId) {
                    window.location.href = `/inventory/${recordId}/edit`;
                }
            }

            function attachDeleteListeners() {
                const deleteButtons = inventorySearchResults.querySelectorAll('.delete-btn');
                deleteButtons.forEach(button => {
                    button.removeEventListener('click', handleDeleteClick);
                    button.addEventListener('click', handleDeleteClick);
                });
            }

            async function handleDeleteClick(event) {
                const recordId = event.target.dataset.id;
                if (!recordId) {
                    showSearchMessage(inventoryTexts.server_error, 'danger');
                    return;
                }

                document.getElementById('deleteModal').style.display = 'flex';
                itemToDelete = recordId;
            }

            document.getElementById('confirm_delete').addEventListener('click', async () => {
                //const secretCode = document.getElementById('secret_code_input').value;
                //if (secretCode !== '1234') {
                //    showSearchMessage("Código secreto incorrecto.", 'danger');
                //    return;
                //}

                try {
                    const response = await fetch('/inventory/delete', {
                        method: 'DELETE',
                        headers: {
                            'Content-Type': 'application/json',
                            'Accept': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')
                                .getAttribute('content')
                        },
                        body: JSON.stringify({
                            id: itemToDelete
                        })
                    });

                    const data = await response.json();

                    if (data.success) {
                        showSearchMessage(inventoryTexts.delete_success, 'success');
                        document.getElementById('deleteModal').style.display = 'none';
                        //  document.getElementById('secret_code_input').value = '';
                        itemToDelete = null;
                        performSearch();
                    } else {
                        showSearchMessage(formatMessage(inventoryTexts.delete_error, {
                            message: data.message || ''
                        }), 'danger');
                    }
                } catch (error) {
                    console.error('Error de comunicación al eliminar:', error);
                    showSearchMessage(inventoryTexts.server_error, 'danger');
                }
            });

            document.getElementById('cancel_delete').addEventListener('click', () => {
                document.getElementById('deleteModal').style.display = 'none';
                // document.getElementById('secret_code_input').value = '';
                itemToDelete = null;
            });

            searchForm.addEventListener('submit', (e) => {
                e.preventDefault();
                currentPage = 1;
                performSearch();
            });

            itemsPerPageSelect.addEventListener('change', () => {
                itemsPerPage = parseInt(itemsPerPageSelect.value);
                currentPage = 1;
                performSearch();
            });

            prevPageBtn.addEventListener('click', () => {
                if (currentPage > 1) {
                    currentPage--;
                    performSearch();
                }
            });

            nextPageBtn.addEventListener('click', () => {
                const totalPages = Math.ceil(totalRecords / itemsPerPage);
                if (currentPage < totalPages) {
                    currentPage++;
                    performSearch();
                }
            });

            // Export button handlers with filter parameters
            function buildExportUrl(baseRoute) {
                const params = new URLSearchParams();

                if (searchBarcode.value.trim()) params.append('barcode', searchBarcode.value.trim());
                if (searchMocaco.value.trim()) params.append('mocaco', searchMocaco.value.trim());
                if (searchNCarton.value.trim()) params.append('box_number', searchNCarton.value.trim());
                if (searchUbicacion.value.trim()) params.append('location', searchUbicacion.value.trim());
                if (searchMarcas.value.trim()) params.append('marcas', searchMarcas.value.trim());
                if (document.getElementById('search_needs_review').checked) params.append('needs_review', '1');

                // Add multiselect filters
                selectedContainers.forEach(container => params.append('containers[]', container));
                selectedSeasons.forEach(season => params.append('seasons[]', season));

                return `${baseRoute}?${params.toString()}`;
            }

            const exportExcelBtn = document.getElementById('export-excel-btn');
            if (exportExcelBtn) {
                exportExcelBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    window.location.href = buildExportUrl("{{ route('inventory.export') }}");
                });
            }

            const exportGroupedBtn = document.getElementById('export-grouped-btn');
            if (exportGroupedBtn) {
                exportGroupedBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    window.location.href = buildExportUrl("{{ route('inventory.export-grouped') }}");
                });
            }

            const exportBrandBtn = document.getElementById('export-brand-btn');
            if (exportBrandBtn) {
                exportBrandBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    const brand = searchMarcas.value.trim();
                    if (!brand) {
                        alert('Por favor, ingresa una Marca en el filtro para poder exportar.');
                        searchMarcas.focus();
                        return;
                    }
                    window.location.href = "{{ route('inventory.export-by-brand') }}?brand=" +
                        encodeURIComponent(brand);
                });
            }

            // Load filter options from API
            async function loadFilterOptions() {
                try {
                    const response = await fetch('/inventory/filter-options', {
                        method: 'GET',
                        headers: {
                            'Accept': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                        }
                    });
                    const data = await response.json();
                    if (data.success) {
                        filterOptions = data;
                        populateMultiselect('containers', data.containers);
                        populateMultiselect('seasons', data.seasons);
                    }
                } catch (error) {
                    console.error('Error loading filter options:', error);
                }
            }

            function populateMultiselect(type, options) {
                const container = document.getElementById(`search_${type}_options`);
                if (options.length === 0) {
                    container.innerHTML = '<div class="text-muted text-center p-2">No hay opciones</div>';
                    return;
                }

                // Store all options for filtering
                container.dataset.allOptions = JSON.stringify(options);

                renderMultiselectOptions(type, options);

                // Add search functionality
                const searchInput = document.getElementById(`${type}_search_input`);
                if (searchInput) {
                    searchInput.addEventListener('input', (e) => {
                        const searchTerm = e.target.value.toLowerCase();
                        const allOptions = JSON.parse(container.dataset.allOptions || '[]');
                        const filteredOptions = allOptions.filter(option =>
                            option.toLowerCase().includes(searchTerm)
                        );
                        renderMultiselectOptions(type, filteredOptions);
                    });

                    // Prevent dropdown from closing when clicking on search input
                    searchInput.addEventListener('click', (e) => {
                        e.stopPropagation();
                    });
                }
            }

            function renderMultiselectOptions(type, options) {
                const container = document.getElementById(`search_${type}_options`);
                container.innerHTML = '';

                if (options.length === 0) {
                    container.innerHTML = '<div class="text-muted text-center p-2">No se encontraron resultados</div>';
                    return;
                }

                // Get currently selected values
                const currentlySelected = type === 'containers' ? selectedContainers : selectedSeasons;

                options.forEach(option => {
                    const div = document.createElement('div');
                    div.className = 'multiselect-option';
                    const isChecked = currentlySelected.includes(option) ? 'checked' : '';
                    div.innerHTML = `
                        <input type="checkbox" value="${option}" id="${type}_${option}" ${isChecked}>
                        <label for="${type}_${option}" style="margin: 0; cursor: pointer;">${option}</label>
                    `;
                    const checkbox = div.querySelector('input');
                    checkbox.addEventListener('change', () => updateMultiselectSelection(type));
                    container.appendChild(div);
                });
            }

            function updateMultiselectSelection(type) {
                const checkboxes = document.querySelectorAll(`#search_${type}_options input[type="checkbox"]:checked`);
                const selected = Array.from(checkboxes).map(cb => cb.value);
                if (type === 'containers') {
                    selectedContainers = selected;
                } else if (type === 'seasons') {
                    selectedSeasons = selected;
                }
                updateMultiselectButton(type, selected.length);
            }

            function updateMultiselectButton(type, count) {
                const btn = document.getElementById(`search_${type}_btn`);
                const textSpan = btn.querySelector('.multiselect-text');
                if (count === 0) {
                    textSpan.innerHTML = 'Seleccionar...';
                } else {
                    textSpan.innerHTML = `Seleccionados <span class="multiselect-count">${count}</span>`;
                }
            }

            // Multiselect dropdown toggle
            function setupMultiselect(type) {
                const btn = document.getElementById(`search_${type}_btn`);
                const dropdown = document.getElementById(`search_${type}_dropdown`);
                const selectAllBtn = document.getElementById(`${type}_select_all`);
                const clearBtn = document.getElementById(`${type}_clear`);

                btn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    dropdown.style.display = dropdown.style.display === 'none' ? 'block' : 'none';
                });

                selectAllBtn.addEventListener('click', () => {
                    const checkboxes = document.querySelectorAll(`#search_${type}_options input[type="checkbox"]`);
                    checkboxes.forEach(cb => cb.checked = true);
                    updateMultiselectSelection(type);
                });

                clearBtn.addEventListener('click', () => {
                    const checkboxes = document.querySelectorAll(`#search_${type}_options input[type="checkbox"]`);
                    checkboxes.forEach(cb => cb.checked = false);
                    updateMultiselectSelection(type);
                });

                // Close dropdown when clicking outside
                document.addEventListener('click', (e) => {
                    if (!btn.contains(e.target) && !dropdown.contains(e.target)) {
                        dropdown.style.display = 'none';
                    }
                });
            }

            // Initialize multiselects
            setupMultiselect('containers');
            setupMultiselect('seasons');
            loadFilterOptions();

            // Carga inicial
            performSearch();
        });
    </script>
@endpush
