@extends('layouts.app')

@section('title', __('client_history.title'))

@section('content')
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2><i class="fas fa-history"></i> {{ __('client_history.header') }}</h2>
    <a href="{{ route('dashboard') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-left"></i> {{ __('client_history.back') }}
    </a>
</div>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-handshake"></i> {{ __('client_history.select_client.title') }}</h5>
            </div>
            <div class="card-body">
                <div class="form-group mb-3">
                    <label for="client_history_select" class="form-label">{{ __('client_history.select_client.label') }}</label>
                    <select id="client_history_select" name="client_history_select" class="form-control" required>
                        <option value="">{{ __('client_history.select_client.loading') }}</option>
                    </select>
                    <div id="client_error" class="text-danger" style="display:none;"></div>
                </div>
            </div>
        </div>

        <div id="client_info_section" class="client-info" style="display:none;">
            <h4 id="client_name_display"></h4>
            <p id="client_email_display"></p>
            <p id="client_phone_display"></p>
        </div>

        <div id="order_history_results" class="mt-4">
            <div class="alert alert-info">
                <i class="fas fa-info-circle"></i> {{ __('client_history.orders.select_client') }}
            </div>
        </div>
    </div>

    <div class="col-md-4">
        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-info-circle"></i> {{ __('client_history.info.title') }}</h5>
            </div>
            <div class="card-body">
                <p class="text-muted">
                    <strong>{{ __('client_history.info.instructions') }}</strong>
                </p>
                <ol class="text-muted small">
                    <li>{{ __('client_history.info.step1') }}</li>
                    <li>{{ __('client_history.info.step2') }}</li>
                    <li>{{ __('client_history.info.step3') }}</li>
                    <li>{{ __('client_history.info.step4') }}</li>
                </ol>
                
                <hr>
                
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i>
                    <strong>{{ __('client_history.info.note') }}</strong> {{ __('client_history.info.note_text') }}
                </div>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-header">
                <h5><i class="fas fa-chart-bar"></i> {{ __('client_history.stats.title') }}</h5>
            </div>
            <div class="card-body">
                <div id="client_stats">
                    <p class="text-muted text-center">{{ __('client_history.stats.select_client') }}</p>
                </div>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-header">
                <h5><i class="fas fa-download"></i> {{ __('client_history.export.title') }}</h5>
            </div>
            <div class="card-body">
                <div id="export_section">
                    <p class="text-muted text-center">{{ __('client_history.export.select_client') }}</p>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<style>
.form-control {
    border: 1px solid var(--sap-border-color);
    border-radius: 4px;
    padding: 10px;
    font-size: 0.9em;
    transition: border-color 0.2s, box-shadow 0.2s;
}

.form-control:focus {
    border-color: var(--sap-blue-light);
    outline: none;
    box-shadow: 0 0 0 2px rgba(52, 152, 219, 0.2);
}

.message {
    padding: 12px;
    margin-bottom: 15px;
    border-radius: 4px;
    font-weight: bold;
    display: block;
    text-align: center;
    font-size: 0.9em;
}

.message.success { 
    background-color: #d4edda; 
    color: #155724; 
    border: 1px solid #c3e6cb; 
}

.message.error { 
    background-color: #f8d7da; 
    color: #721c24; 
    border: 1px solid #f5c6cb; 
}

.message.info { 
    background-color: #d1ecf1; 
    color: #0c5460; 
    border: 1px solid #bee5eb; 
}

.order-item {
    background-color: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 4px;
    padding: 15px;
    margin-bottom: 15px;
    transition: all 0.2s ease;
}

.order-item:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.order-item h6 {
    margin-bottom: 8px;
    color: #495057;
}

.order-item small {
    color: #6c757d;
}

.stats-card {
    background: white;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    margin-bottom: 20px;
}

.stats-icon {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    background: linear-gradient(135deg, var(--sap-blue-light), #2980b9);
    display: flex;
    align-items: center;
    justify-content: center;
    margin-right: 15px;
    color: white;
    font-size: 1.5em;
}

.stats-content h3 {
    font-size: 2em;
    font-weight: bold;
    color: var(--sap-blue-dark);
    margin: 0;
    line-height: 1;
}

.stats-content p {
    color: var(--sap-text-medium);
    margin: 5px 0 0 0;
    font-size: 0.9em;
}

.table-responsive {
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.table {
    margin-bottom: 0;
}

.table th {
    background-color: #f8f9fa;
    border-top: none;
    font-weight: 600;
    color: var(--sap-text-medium);
    font-size: 0.85em;
    padding: 12px;
}

.table td {
    padding: 12px;
    font-size: 0.85em;
    vertical-align: middle;
    border-top: 1px solid #dee2e6;
}

.table tbody tr:hover {
    background-color: #f8f9fa;
}

.badge {
    padding: 4px 8px;
    font-size: 0.75em;
    font-weight: 500;
    border-radius: 4px;
}

.badge-success {
    background-color: #28a745;
    color: white;
}

.btn-sm {
    padding: 5px 10px;
    font-size: 0.8em;
    min-width: auto;
}

.client-info {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
}

.client-info h4 {
    margin-bottom: 10px;
}

.client-info p {
    margin-bottom: 5px;
    opacity: 0.9;
}

/* Modal Styles */
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: rgba(0, 0, 0, 0.7);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
    padding: 20px;
}

.modal-container {
    background: white;
    border-radius: 8px;
    max-width: 800px;
    width: 100%;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px;
    border-bottom: 1px solid #dee2e6;
}

.modal-title {
    margin: 0;
    font-size: 1.25em;
    font-weight: 600;
}

.modal-close {
    background: none;
    border: none;
    font-size: 2em;
    cursor: pointer;
    color: #6c757d;
    line-height: 1;
    padding: 0;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.modal-close:hover {
    color: #000;
}

.modal-body {
    padding: 20px;
}

.modal-footer {
    padding: 15px 20px;
    border-top: 1px solid #dee2e6;
    display: flex;
    justify-content: flex-end;
    gap: 10px;
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
@php
$clientHistoryTranslations = [
    'select_placeholder' => __('client_history.select_client.placeholder'),
    'select_client_option' => __('client_history.select_client.select_placeholder'),
    'no_clients' => __('client_history.select_client.no_clients'),
    'clients_load_error' => __('client_history.messages.clients_load_error'),
    'network_error' => __('client_history.messages.network_error'),
    'select_client_message' => __('client_history.orders.select_client'),
    'loading_orders' => __('client_history.orders.loading'),
    'no_orders' => __('client_history.orders.no_orders'),
    'orders_error' => __('client_history.orders.error'),
    'orders_title' => __('client_history.orders.title'),
    'order_number' => __('client_history.orders.order_number'),
    'order_date' => __('client_history.orders.order_date'),
    'dispatch_date' => __('client_history.orders.dispatch_date'),
    'operator' => __('client_history.orders.operator'),
    'items' => __('client_history.orders.items'),
    'notes' => __('client_history.orders.notes'),
    'view_details' => __('client_history.orders.view_details'),
    'stats_select_client' => __('client_history.stats.select_client'),
    'total_orders' => __('client_history.stats.total_orders'),
    'total_items' => __('client_history.stats.total_items'),
    'total_boxes' => __('client_history.stats.total_boxes'),
    'last_order' => __('client_history.stats.last_order'),
    'stats_error' => __('client_history.stats.error'),
    'export_select_client' => __('client_history.export.select_client'),
    'export_button' => __('client_history.export.button'),
    'email' => __('client_history.messages.email'),
    'phone' => __('client_history.messages.phone'),
    'not_available' => __('client_history.messages.not_available'),
    'order_details_title' => __('client_history.order_details.title'),
    'order_details_client' => __('client_history.order_details.client'),
    'order_details_order_date' => __('client_history.order_details.order_date'),
    'order_details_dispatch_date' => __('client_history.order_details.dispatch_date'),
    'order_details_operator' => __('client_history.order_details.operator'),
    'order_details_status' => __('client_history.order_details.status'),
    'order_details_status_dispatched' => __('client_history.order_details.status_dispatched'),
    'order_details_total_items' => __('client_history.order_details.total_items'),
    'order_details_notes' => __('client_history.order_details.notes'),
    'order_details_barcode' => __('client_history.order_details.barcode'),
    'order_details_mocaco' => __('client_history.order_details.mocaco'),
    'order_details_box_number' => __('client_history.order_details.box_number'),
    'order_details_close' => __('client_history.order_details.close'),
    'order_details_error' => __('client_history.order_details.error'),
];
@endphp
const translations = @json($clientHistoryTranslations);

// Función helper para formatear mensajes con parámetros
function formatMessage(key, params = {}) {
    let message = translations[key] || key;
    Object.keys(params).forEach(param => {
        message = message.replace(`:${param}`, params[param]);
    });
    return message;
}

document.addEventListener('DOMContentLoaded', function() {
    const clientHistorySelect = document.getElementById('client_history_select');
    const orderHistoryResults = document.getElementById('order_history_results');
    const clientInfoSection = document.getElementById('client_info_section');
    const clientStatsDiv = document.getElementById('client_stats');
    const exportSectionDiv = document.getElementById('export_section');

    // Initialize Select2
    $('#client_history_select').select2({
        placeholder: translations.select_placeholder,
        allowClear: true,
        width: '100%'
    });

    // Listen for Select2 change event
    $('#client_history_select').on('change', async function() {
        const clientId = this.value;
        console.log('Client selected (Select2):', clientId);
        
        if (clientId) {
            console.log('Loading data for client:', clientId);
            await Promise.all([
                loadClientOrders(clientId),
                loadClientStats(clientId),
                loadClientInfo(clientId)
            ]);
        } else {
            orderHistoryResults.innerHTML = `
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> ${translations.select_client_message}
                </div>
            `;
            clientInfoSection.style.display = 'none';
            clientStatsDiv.innerHTML = `<p class="text-muted text-center">${translations.stats_select_client}</p>`;
            exportSectionDiv.innerHTML = `<p class="text-muted text-center">${translations.export_select_client}</p>`;
        }
    });

    function showMessage(msg, type) {
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${type}`;
        messageDiv.textContent = msg;
        messageDiv.style.display = 'block';
        
        orderHistoryResults.innerHTML = '';
        orderHistoryResults.appendChild(messageDiv);
        
        setTimeout(() => {
            messageDiv.style.display = 'none';
        }, 5000);
    }

    async function loadClients() {
        try {
            const response = await fetch('/api/client-history/clients');
            const clients = await response.json();
            
            $('#client_history_select').empty().append($(`<option value="">${translations.select_client_option}</option>`)).trigger('change');

            if (clients.length > 0) {
                console.log('Clients loaded:', clients);
                clients.forEach(client => {
                    const newOption = new Option(client.client_name, client.id, false, false);
                    $('#client_history_select').append(newOption);
                    console.log('Added client:', client.client_name, 'ID:', client.id);
                });
                $('#client_history_select').trigger('change');
            } else {
                $('#client_history_select').append($(`<option value="">${translations.no_clients}</option>`)).trigger('change');
                showMessage(translations.clients_load_error, 'error');
            }
        } catch (error) {
            console.error('Error al cargar clientes:', error);
            showMessage(translations.network_error, 'error');
        }
    }

    async function loadClientOrders(clientId) {
        try {
            console.log('Loading orders for client:', clientId);
            showMessage(translations.loading_orders, 'info');
            
            const response = await fetch(`/api/client-history/orders/${clientId}`);
            console.log('Response status:', response.status);
            const orders = await response.json();
            console.log('Orders received:', orders);

            if (orders.length > 0) {
                let html = `<h5><i class="fas fa-list-alt"></i> ${translations.orders_title}</h5>`;
                
                orders.forEach(order => {
                    const orderDate = new Date(order.order_date).toLocaleDateString();
                    const dispatchDate = order.dispatched_date ? new Date(order.dispatched_date).toLocaleDateString() : translations.not_available;
                    const itemsCount = order.order_items ? order.order_items.length : 0;
                    
                    html += `
                        <div class="order-item">
                            <div class="d-flex justify-content-between align-items-start">
                                <div>
                                    <h6><i class="fas fa-shopping-cart"></i> ${formatMessage('order_number', { id: order.id })}</h6>
                                    <small><strong>${translations.order_date}</strong> ${orderDate}</small><br>
                                    <small><strong>${translations.dispatch_date}</strong> ${dispatchDate}</small><br>
                                    <small><strong>${translations.operator}</strong> ${order.operario_id}</small><br>
                                    <small><strong>${translations.items}</strong> ${itemsCount}</small>
                                    ${order.notes ? `<br><small><strong>${translations.notes}</strong> ${order.notes}</small>` : ''}
                                </div>
                                <div>
                                    <button class="btn btn-sm btn-outline-primary" onclick="showOrderDetails(${order.id})">
                                        <i class="fas fa-eye"></i> ${translations.view_details}
                                    </button>
                                </div>
                            </div>
                        </div>
                    `;
                });
                
                orderHistoryResults.innerHTML = html;
            } else {
                orderHistoryResults.innerHTML = `
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i> ${translations.no_orders}
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error al cargar pedidos:', error);
            showMessage(translations.orders_error, 'error');
        }
    }

    async function loadClientStats(clientId) {
        try {
            const response = await fetch(`/api/client-history/stats/${clientId}`);
            const stats = await response.json();
            
            const lastOrderDate = stats.last_order_date ? new Date(stats.last_order_date).toLocaleDateString() : 'N/A';
            
            clientStatsDiv.innerHTML = `
                <div class="stats-card d-flex align-items-center">
                    <div class="stats-icon">
                        <i class="fas fa-shopping-cart"></i>
                    </div>
                    <div class="stats-content">
                        <h3>${stats.total_orders}</h3>
                        <p>${translations.total_orders}</p>
                    </div>
                </div>
                
                <div class="stats-card d-flex align-items-center">
                    <div class="stats-icon">
                        <i class="fas fa-boxes"></i>
                    </div>
                    <div class="stats-content">
                        <h3>${stats.total_items}</h3>
                        <p>${translations.total_items}</p>
                    </div>
                </div>
                
                <div class="stats-card d-flex align-items-center">
                    <div class="stats-icon">
                        <i class="fas fa-archive"></i>
                    </div>
                    <div class="stats-content">
                        <h3>${stats.total_boxes}</h3>
                        <p>${translations.total_boxes}</p>
                    </div>
                </div>
                
                <div class="alert alert-info">
                    <small><strong>${translations.last_order}</strong> ${lastOrderDate}</small>
                </div>
            `;
        } catch (error) {
            console.error('Error al cargar estadísticas:', error);
            clientStatsDiv.innerHTML = `<p class="text-muted text-center">${translations.stats_error}</p>`;
        }
    }

    async function loadClientInfo(clientId) {
        try {
            const response = await fetch(`/api/client-history/clients`);
            const clients = await response.json();
            
            const client = clients.find(c => c.id == clientId);
            
            if (client) {
                document.getElementById('client_name_display').textContent = client.client_name;
                document.getElementById('client_email_display').textContent = `${translations.email} ${client.email || translations.not_available}`;
                document.getElementById('client_phone_display').textContent = `${translations.phone} ${client.phone || translations.not_available}`;
                
                clientInfoSection.style.display = 'block';
            }
            
            // Update export section
            exportSectionDiv.innerHTML = `
                <button class="btn btn-success w-100" onclick="exportClientHistory(${clientId})">
                    <i class="fas fa-download"></i> ${translations.export_button}
                </button>
            `;
        } catch (error) {
            console.error('Error al cargar información del cliente:', error);
        }
    }

    // Global functions for order details and export
    window.showOrderDetails = async function(orderId) {
        try {
            const response = await fetch(`/api/client-history/order-details/${orderId}`);
            const order = await response.json();
            
            let itemsHtml = '';
            if (order.order_items && order.order_items.length > 0) {
                itemsHtml = `
                    <div class="table-responsive mt-3">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>${translations.order_details_barcode}</th>
                                    <th>${translations.order_details_mocaco}</th>
                                    <th>${translations.order_details_box_number}</th>
                                </tr>
                            </thead>
                            <tbody>
                `;
                
                order.order_items.forEach(item => {
                    itemsHtml += `
                        <tr>
                            <td><code>${item.full_barcode}</code></td>
                            <td>${item.mocaco}</td>
                            <td>${item.n_carton}</td>
                        </tr>
                    `;
                });
                
                itemsHtml += `
                            </tbody>
                        </table>
                    </div>
                `;
            }
            
            const modalHtml = `
                <div class="modal-overlay" id="orderDetailsModal" onclick="this.remove()">
                    <div class="modal-container" onclick="event.stopPropagation()">
                        <div class="modal-header">
                            <h5 class="modal-title">${formatMessage('order_details_title', { id: order.id })}</h5>
                            <button type="button" class="modal-close" onclick="document.getElementById('orderDetailsModal').remove()">×</button>
                        </div>
                        <div class="modal-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <p><strong>${translations.order_details_client}</strong> ${order.client.client_name}</p>
                                        <p><strong>${translations.order_details_order_date}</strong> ${new Date(order.order_date).toLocaleDateString()}</p>
                                        <p><strong>${translations.order_details_dispatch_date}</strong> ${order.dispatched_date ? new Date(order.dispatched_date).toLocaleDateString() : translations.not_available}</p>
                                        <p><strong>${translations.order_details_operator}</strong> ${order.operario_id}</p>
                                    </div>
                                    <div class="col-md-6">
                                        <p><strong>${translations.order_details_status}</strong> <span class="badge badge-success">${translations.order_details_status_dispatched}</span></p>
                                        <p><strong>${translations.order_details_total_items}</strong> ${order.order_items ? order.order_items.length : 0}</p>
                                        ${order.notes ? `<p><strong>${translations.order_details_notes}</strong> ${order.notes}</p>` : ''}
                                    </div>
                                </div>
                                ${itemsHtml}
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" onclick="this.closest('.modal-overlay').remove()">${translations.order_details_close}</button>
                            </div>
                    </div>
                </div>
            `;
            
            // Remove existing modal if any
            const existingModal = document.getElementById('orderDetailsModal');
            if (existingModal) {
                existingModal.remove();
            }
            
            // Add modal to body
            document.body.insertAdjacentHTML('beforeend', modalHtml);
            
            // No need for manual show - CSS handles it
        } catch (error) {
            console.error('Error al cargar detalles del pedido:', error);
            showMessage(translations.order_details_error, 'error');
        }
    };

    window.exportClientHistory = function(clientId) {
        window.open(`/api/client-history/export/${clientId}`, '_blank');
    };

    // Initialize
    loadClients();
});
</script>
@endpush
