@extends('layouts.app')

@section('title', __('articles.title'))

@section('content')
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2><i class="fas fa-tags"></i> {{ __('articles.title') }}</h2>
    <div class="d-flex gap-2">
        @can('articles.create')
        <a href="{{ route('articles.import-sample') }}" class="btn btn-secondary">
            <i class="fas fa-download"></i> {{ __('articles.buttons.sample_csv') }}
        </a>
        <form action="{{ route('articles.import') }}" method="POST" enctype="multipart/form-data" class="d-inline-block">
            @csrf
            <div class="input-group">
                <input type="file" name="file" accept=".csv" class="form-control" required>
                <button type="submit" class="btn">
                    <i class="fas fa-file-import"></i> {{ __('articles.buttons.import_csv') }}
                </button>
            </div>
        </form>
        <a href="{{ route('articles.create') }}" class="btn">
            <i class="fas fa-plus"></i> {{ __('articles.buttons.add_article') }}
        </a>
        @endcan
    </div>
</div>

@if(session('success'))
<div class="alert alert-success alert-dismissible fade show mb-4" role="alert">
    <i class="fas fa-check-circle"></i> {!! nl2br(e(session('success'))) !!}
    @if(session('imported_count'))
    <br><strong>{{ __('articles.alerts.imported') }}</strong> {{ session('imported_count') }}
    @endif
    @if(session('rejected_count'))
    <br><strong>{{ __('articles.alerts.rejected') }}</strong> {{ session('rejected_count') }}
    @endif
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
@endif

@if(session('error'))
<div class="alert alert-danger alert-dismissible fade show mb-4" role="alert">
    <i class="fas fa-exclamation-circle"></i> {!! nl2br(e(session('error'))) !!}
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
@endif

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-header">
        <h5><i class="fas fa-filter"></i> {{ __('articles.filters.title') }}</h5>
    </div>
    <div class="card-body">
        <form method="GET" action="{{ route('articles.index') }}" class="row g-3">
            <div class="col-md-3">
                <label for="search" class="form-label">{{ __('articles.filters.search') }}</label>
                <input type="text" class="form-control" id="search" name="search" 
                       value="{{ request('search') }}" placeholder="{{ __('articles.filters.search_placeholder') }}">
            </div>
            <div class="col-md-3">
                <label for="family" class="form-label">{{ __('articles.filters.family') }}</label>
                <input type="text" class="form-control" id="family" name="family" 
                       value="{{ request('family') }}" placeholder="{{ __('articles.filters.family') }}">
            </div>
            <div class="col-md-3">
                <label for="brand" class="form-label">{{ __('articles.filters.brand') }}</label>
                <input type="text" class="form-control" id="brand" name="brand" 
                       value="{{ request('brand') }}" placeholder="{{ __('articles.filters.brand') }}">
            </div>
            <div class="col-md-3">
                <label class="form-label">&nbsp;</label>
                <div class="d-flex gap-2">
                    <button type="submit" class="btn">
                        <i class="fas fa-search"></i> {{ __('articles.filters.apply') }}
                    </button>
                    <a href="{{ route('articles.index') }}" class="btn btn-secondary">
                        <i class="fas fa-times"></i> {{ __('articles.filters.reset') }}
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Controles de Paginación -->
@if($articles->total() > 0 || $articles->hasPages())
<div class="card mb-3">
    <div class="card-body">
        <form method="GET" action="{{ route('articles.index') }}" id="pagination-form" class="d-flex justify-content-between align-items-center flex-wrap gap-3">
            <!-- Mantener filtros en campos ocultos -->
            @if(request('search'))
                <input type="hidden" name="search" value="{{ request('search') }}">
            @endif
            @if(request('family'))
                <input type="hidden" name="family" value="{{ request('family') }}">
            @endif
            @if(request('brand'))
                <input type="hidden" name="brand" value="{{ request('brand') }}">
            @endif
            
            <div class="d-flex align-items-center gap-2">
                <label for="per_page">{{ __('articles.pagination.rows_per_page') }}</label>
                <select name="per_page" id="per_page" class="form-control" style="width: auto;" onchange="this.form.submit()">
                    <option value="10" {{ request('per_page', 20) == 10 ? 'selected' : '' }}>10</option>
                    <option value="25" {{ request('per_page', 20) == 25 ? 'selected' : '' }}>25</option>
                    <option value="50" {{ request('per_page', 20) == 50 ? 'selected' : '' }}>50</option>
                    <option value="100" {{ request('per_page', 20) == 100 ? 'selected' : '' }}>100</option>
                    <option value="250" {{ request('per_page', 20) == 250 ? 'selected' : '' }}>250</option>
                </select>
            </div>
            <div class="text-muted">
                {{ __('articles.pagination.page_info', [
                    'current' => $articles->currentPage(),
                    'total' => $articles->lastPage(),
                    'records' => $articles->total()
                ]) }}
            </div>
            <div class="d-flex align-items-center gap-2">
                @if($articles->onFirstPage())
                    <button type="button" class="btn btn-sm" disabled>{{ __('articles.pagination.previous') }}</button>
                @else
                    <a href="{{ $articles->previousPageUrl() }}" class="btn btn-sm">{{ __('articles.pagination.previous') }}</a>
                @endif
                
                @foreach($articles->getUrlRange(max(1, $articles->currentPage() - 2), min($articles->lastPage(), $articles->currentPage() + 2)) as $page => $url)
                    @if($page == $articles->currentPage())
                        <span class="pagination-link active">{{ $page }}</span>
                    @else
                        <a href="{{ $url }}" class="pagination-link">{{ $page }}</a>
                    @endif
                @endforeach
                
                @if($articles->hasMorePages())
                    <a href="{{ $articles->nextPageUrl() }}" class="btn btn-sm">{{ __('articles.pagination.next') }}</a>
                @else
                    <button type="button" class="btn btn-sm" disabled>{{ __('articles.pagination.next') }}</button>
                @endif
            </div>
        </form>
    </div>
</div>
@endif

<!-- Tabla de Artículos -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5><i class="fas fa-list"></i> {{ __('articles.table.title') }}</h5>
        <span class="badge bg-primary">{{ __('articles.table.count', ['count' => $articles->total()]) }}</span>
    </div>
    <div class="card-body p-0">
        @if($articles->count() > 0)
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>{{ __('articles.table.columns.id') }}</th>
                            <th>{{ __('articles.table.columns.mocaco') }}</th>
                            <th>{{ __('articles.table.columns.family') }}</th>
                            <th>{{ __('articles.table.columns.brand') }}</th>
                            <th>{{ __('articles.table.columns.price') }}</th>
                            <th>{{ __('articles.table.columns.weight') }}</th>
                            <th>{{ __('articles.table.columns.campaign') }}</th>
                            <th>{{ __('articles.table.columns.actions') }}</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($articles as $article)
                            <tr>
                                <td>{{ $article->id }}</td>
                                <td><code>{{ $article->mocaco }}</code></td>
                                <td>{{ $article->familia ?? '-' }}</td>
                                <td>{{ $article->cadena ?? '-' }}</td>
                                <td>{{ $article->precio_pvp_maximo_temporada ? '€' . number_format($article->precio_pvp_maximo_temporada, 2) : '-' }}</td>
                                <td>{{ $article->peso_unitario ? $article->peso_unitario . ' kg' : '-' }}</td>
                                <td>{{ $article->campana ?? '-' }}</td>
                                <td>
                                    <div class="btn-group" role="group">
                                        <a href="{{ route('articles.show', $article) }}" class="btn btn-sm btn-outline-info">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        @can('articles.update')
                                        <a href="{{ route('articles.edit', $article) }}" class="btn btn-sm btn-outline-warning">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        @endcan
                                        @can('articles.delete')
                                        <form action="{{ route('articles.destroy', $article) }}" method="POST" 
                                              class="d-inline" onsubmit="return confirm('{{ __('articles.confirm_delete') }}')">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit" class="btn btn-sm btn-outline-danger">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                        @endcan
                                    </div>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
        @else
            <div class="text-center py-5">
                <i class="fas fa-tags fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">{{ __('articles.table.empty.title') }}</h5>
                <p class="text-muted">{{ __('articles.table.empty.description') }}</p>
                <a href="{{ route('articles.create') }}" class="btn">
                    <i class="fas fa-plus"></i> {{ __('articles.buttons.add_first_article') }}
                </a>
            </div>
        @endif
    </div>
</div>

<!-- Mostrar artículos rechazados si existen -->
@if(session('rejected_items') && count(session('rejected_items')) > 0)
<div class="card mt-4 border-warning">
    <div class="card-header bg-warning text-dark">
        <h5><i class="fas fa-exclamation-triangle"></i> {{ __('articles.rejected.title', ['count' => session('rejected_count')]) }}</h5>
    </div>
    <div class="card-body">
        <p class="text-muted">{{ __('articles.rejected.description') }}</p>
        <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
            <table class="table table-sm table-hover">
                <thead class="table-light sticky-top">
                    <tr>
                        <th>{{ __('articles.rejected.columns.line') }}</th>
                        <th>{{ __('articles.rejected.columns.mocaco') }}</th>
                        <th>{{ __('articles.rejected.columns.campaign') }}</th>
                        <th>{{ __('articles.rejected.columns.family_new') }}</th>
                        <th>{{ __('articles.rejected.columns.composition_new') }}</th>
                        <th>{{ __('articles.rejected.columns.existing_article') }}</th>
                        <th>{{ __('articles.rejected.columns.family_existing') }}</th>
                        <th>{{ __('articles.rejected.columns.composition_existing') }}</th>
                        <th>{{ __('articles.rejected.columns.created_at') }}</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach(session('rejected_items') as $rejected)
                    <tr class="table-warning">
                        <td><strong>{{ $rejected['line'] }}</strong></td>
                        <td><code>{{ $rejected['mocaco'] }}</code></td>
                        <td>{{ $rejected['campana'] }}</td>
                        <td>{{ $rejected['familia'] ?: '-' }}</td>
                        <td>{{ $rejected['composition'] ?: '-' }}</td>
                        <td>
                            <a href="{{ route('articles.show', $rejected['existing_id']) }}" target="_blank" class="btn btn-sm btn-outline-info">
                                {{ __('articles.rejected.view_article', ['id' => $rejected['existing_id']]) }}
                            </a>
                        </td>
                        <td>{{ $rejected['existing_familia'] ?: '-' }}</td>
                        <td>{{ $rejected['existing_composition'] ?: '-' }}</td>
                        <td>{{ $rejected['existing_created_at'] ?? '-' }}</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
</div>
@endif
@endsection

@push('styles')
<style>
.btn-group {
    display: flex;
    gap: 5px;
}

.btn-sm {
    padding: 5px 10px;
    font-size: 0.8em;
    min-width: auto;
}

.btn-outline-info {
    background-color: transparent;
    color: #17a2b8;
    border: 1px solid #17a2b8;
}

.btn-outline-info:hover {
    background-color: #17a2b8;
    color: white;
}

.btn-outline-warning {
    background-color: transparent;
    color: #ffc107;
    border: 1px solid #ffc107;
}

.btn-outline-warning:hover {
    background-color: #ffc107;
    color: white;
}

.btn-outline-danger {
    background-color: transparent;
    color: #dc3545;
    border: 1px solid #dc3545;
}

.btn-outline-danger:hover {
    background-color: #dc3545;
    color: white;
}

.badge {
    padding: 0.5em 0.75em;
    font-size: 0.75em;
    font-weight: 700;
    line-height: 1;
    text-align: center;
    white-space: nowrap;
    vertical-align: baseline;
    border-radius: 0.375rem;
}

.bg-primary {
    background-color: var(--sap-blue-light) !important;
}

code {
    font-family: 'Courier New', monospace;
    background-color: #f8f9fa;
    padding: 2px 4px;
    border-radius: 3px;
    font-size: 0.9em;
}

/* Asegurar que la tabla tenga los mismos estilos que inventario */
.data-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 0;
    font-size: 13px;
    box-shadow: 0 1px 5px rgba(0,0,0,0.08);
    border-radius: 0;
    overflow: hidden;
}

.data-table th, .data-table td {
    border: 1px solid #DFE3E8;
    padding: 8px 12px;
    text-align: left;
    vertical-align: middle;
}

.data-table th {
    background-color: #F7F9FC;
    color: #555;
    font-weight: 600;
    position: relative;
    cursor: default;
    padding-right: 25px;
    user-select: none;
    white-space: nowrap;
    transition: background-color 0.2s ease;
}

.data-table th:hover {
    background-color: #E6EEF5;
}

.data-table tbody tr:nth-child(even) {
    background-color: #FCFDFF;
}

.data-table tbody tr:hover {
    background-color: #EBF2F9;
}

.data-table tbody tr td {
    border: 1px solid #DFE3E8;
}

.table-responsive {
    overflow-x: auto;
}

.pagination-link {
    padding: 5px 10px;
    margin: 0 2px;
    border: 1px solid #007bff;
    border-radius: 3px;
    text-decoration: none;
    color: #007bff;
    cursor: pointer;
    font-size: 14px;
    display: inline-block;
}

.pagination-link:hover:not(.active) {
    background-color: #e6f2ff;
}

.pagination-link.active {
    background-color: #007bff;
    color: white;
    pointer-events: none;
}
</style>
@endpush

