<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class AddManageInvoicesPermissionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * Este seeder agrega el permiso 'orders.manage_invoices' y lo asigna al rol 'facturacion'
     * Útil para actualizaciones en producción sin ejecutar todo el InitialDataSeeder
     */
    public function run(): void
    {
        // Crear el permiso si no existe
        $permission = Permission::firstOrCreate(
            ['name' => 'orders.manage_invoices'],
            [
                'guard_name' => 'web',
            ]
        );

        $this->command->info("Permiso 'orders.manage_invoices' creado/verificado.");

        // Asignar el permiso al rol 'facturacion' si existe
        $facturacionRole = Role::where('name', 'facturacion')->first();
        
        if ($facturacionRole) {
            if (!$facturacionRole->hasPermissionTo($permission)) {
                $facturacionRole->givePermissionTo($permission);
                $this->command->info("Permiso asignado al rol 'facturacion'.");
            } else {
                $this->command->info("El rol 'facturacion' ya tenía el permiso asignado.");
            }
        } else {
            $this->command->warn("El rol 'facturacion' no existe. Creando el rol...");
            
            // Crear el rol si no existe
            $facturacionRole = Role::firstOrCreate(
                ['name' => 'facturacion'],
                [
                    'guard_name' => 'web',
                ]
            );
            
            // Verificar y crear los permisos necesarios si no existen
            $permissionsToAssign = [];
            
            // Verificar orders.view
            $ordersViewPermission = Permission::firstOrCreate(
                ['name' => 'orders.view'],
                ['guard_name' => 'web']
            );
            $permissionsToAssign[] = $ordersViewPermission;
            
            // Agregar el permiso principal
            $permissionsToAssign[] = $permission;
            
            // Verificar clients.view
            $clientsViewPermission = Permission::firstOrCreate(
                ['name' => 'clients.view'],
                ['guard_name' => 'web']
            );
            $permissionsToAssign[] = $clientsViewPermission;
            
            // Asignar todos los permisos
            $facturacionRole->syncPermissions($permissionsToAssign);
            
            $this->command->info("Rol 'facturacion' creado con los permisos necesarios.");
        }

        // También asegurarse de que el rol 'admin' tenga el permiso (por si acaso)
        $adminRole = Role::where('name', 'admin')->first();
        if ($adminRole) {
            // El admin ya debería tener todos los permisos, pero por si acaso
            if (!$adminRole->hasPermissionTo($permission)) {
                $adminRole->givePermissionTo($permission);
                $this->command->info("Permiso también asignado al rol 'admin'.");
            }
        }

        $this->command->info('Seeder completado exitosamente.');
    }
}
