<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Inventory;
use App\Models\AuditLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class OrderService
{
    /**
     * Create draft order with inventory items.
     */
    public function createDraftOrder(array $data): array
    {
        try {
            DB::beginTransaction();

            // Create order
            $order = Order::create([
                'client_id' => $data['client_id'],
                'user_id' => Auth::id(),
                'operario_id' => $data['operario_id'] ?? null,
                'notes' => $data['notes'] ?? null,
                'status' => 'draft',
                'order_date' => now(),
            ]);

            // Get inventory items for each box
            foreach ($data['boxes'] as $boxNumber) {
                $inventoryItems = Inventory::byBox($boxNumber)->get();
                
                foreach ($inventoryItems as $item) {
                    OrderItem::create([
                        'order_id' => $order->id,
                        'inventory_id' => $item->id,
                        'full_barcode' => $item->full_barcode,
                        'mocaco' => $item->mocaco,
                        'n_carton' => $item->n_carton,
                    ]);
                }
            }

            // Log the action
            AuditLog::logAction(
                'create_order',
                $order,
                Auth::user(),
                [],
                $order->toArray(),
                "Borrador de pedido creado con {$order->orderItems()->count()} artículos"
            );

            DB::commit();

            return [
                'success' => true,
                'order_id' => $order->id,
                'message' => 'Borrador guardado con éxito.',
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            
            return [
                'success' => false,
                'message' => 'Error al guardar el borrador: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Confirm and dispatch order.
     */
    public function confirmDispatch(int $orderId): array
    {
        try {
            DB::beginTransaction();

            $order = Order::findOrFail($orderId);

            if ($order->status !== 'draft') {
                return [
                    'success' => false,
                    'message' => 'Solo se pueden despachar pedidos en estado borrador.',
                ];
            }

            // Get all inventory items in this order
            $orderItems = $order->orderItems()->with('inventory')->get();
            $inventoryIds = $orderItems->pluck('inventory_id')->toArray();

            // Delete inventory items (they are being dispatched)
            $deletedCount = Inventory::whereIn('id', $inventoryIds)->delete();

            if ($deletedCount !== count($inventoryIds)) {
                throw new \Exception('No se pudieron eliminar todos los artículos del inventario.');
            }

            // Update order status
            $order->dispatch();

            // Log the action
            AuditLog::logAction(
                'dispatch_order',
                $order,
                Auth::user(),
                ['status' => 'draft'],
                ['status' => 'dispatched', 'dispatched_date' => now()],
                "Pedido despachado con {$deletedCount} artículos"
            );

            DB::commit();

            return [
                'success' => true,
                'message' => 'Pedido confirmado y enviado con éxito.',
                'dispatched_items' => $deletedCount,
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            
            return [
                'success' => false,
                'message' => 'Error al confirmar el despacho: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Cancel order.
     */
    public function cancelOrder(int $orderId): array
    {
        try {
            DB::beginTransaction();

            $order = Order::findOrFail($orderId);

            if ($order->status === 'dispatched') {
                return [
                    'success' => false,
                    'message' => 'No se puede cancelar un pedido ya despachado.',
                ];
            }

            $oldStatus = $order->status;
            $order->update(['status' => 'cancelled']);

            // Log the action
            AuditLog::logAction(
                'cancel_order',
                $order,
                Auth::user(),
                ['status' => $oldStatus],
                ['status' => 'cancelled'],
                "Pedido cancelado"
            );

            DB::commit();

            return [
                'success' => true,
                'message' => 'Pedido cancelado correctamente.',
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            
            return [
                'success' => false,
                'message' => 'Error al cancelar el pedido: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get order statistics.
     */
    public function getOrderStatistics(): array
    {
        $stats = [
            'total_orders' => Order::count(),
            'draft_orders' => Order::draft()->count(),
            'dispatched_orders' => Order::dispatched()->count(),
            'orders_by_status' => Order::selectRaw('status, COUNT(*) as count')
                ->groupBy('status')
                ->get(),
            'orders_by_client' => Order::with('client')
                ->selectRaw('client_id, COUNT(*) as count')
                ->groupBy('client_id')
                ->orderBy('count', 'desc')
                ->limit(10)
                ->get(),
            'recent_orders' => Order::with(['client', 'user'])
                ->orderBy('order_date', 'desc')
                ->limit(10)
                ->get(),
        ];

        return [
            'success' => true,
            'statistics' => $stats,
        ];
    }
}










