<?php

namespace App\Services;

use App\Models\ContainerEntry;
use App\Models\TheoreticalArticle;
use App\Models\Inventory;
use App\Enums\ContainerEntryStatus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ContainerEntryService
{
    /**
     * Calcular el porcentaje de avance del escaneo
     * 
     * @param ContainerEntry $containerEntry
     * @return array
     */
    public function calculateProgress(ContainerEntry $containerEntry): array
    {
        // Unidades teóricas totales
        $theoreticalUnits = TheoreticalArticle::where('container_entry_id', $containerEntry->id)
            ->sum('cantidad_unidades');

        // Unidades escaneadas (basado en inventory con el mismo contenedor)
        $scannedUnits = Inventory::where('conteneur', $containerEntry->n_camion)
            ->count();

        $percentage = $theoreticalUnits > 0 
            ? round(($scannedUnits / $theoreticalUnits) * 100, 2) 
            : 0;

        return [
            'theoretical_units' => $theoreticalUnits,
            'scanned_units' => $scannedUnits,
            'percentage' => $percentage,
            'remaining_units' => max(0, $theoreticalUnits - $scannedUnits),
        ];
    }

    /**
     * Generar reporte de discrepancias
     * 
     * @param ContainerEntry $containerEntry
     * @return array
     */
    public function generateDiscrepancyReport(ContainerEntry $containerEntry): array
    {
        $discrepancies = [];
        
        // Obtener todos los artículos teóricos agrupados por MOCACO
        $theoretical = TheoreticalArticle::where('container_entry_id', $containerEntry->id)
            ->select('mocaco', DB::raw('SUM(cantidad_unidades) as unidades_teoricas'))
            ->groupBy('mocaco')
            ->get()
            ->keyBy('mocaco');

        // Obtener unidades escaneadas por MOCACO del mismo contenedor
        $scanned = Inventory::where('conteneur', $containerEntry->n_camion)
            ->select('mocaco', DB::raw('COUNT(*) as unidades_escaneadas'))
            ->groupBy('mocaco')
            ->get()
            ->keyBy('mocaco');

        // Comparar teóricos vs escaneados
        foreach ($theoretical as $mocaco => $theoreticalItem) {
            $scannedCount = $scanned[$mocaco]->unidades_escaneadas ?? 0;
            $theoreticalCount = $theoreticalItem->unidades_teoricas;
            
            if ($scannedCount != $theoreticalCount) {
                $discrepancies[] = [
                    'mocaco' => $mocaco,
                    'unidades_teoricas' => $theoreticalCount,
                    'unidades_escaneadas' => $scannedCount,
                    'diferencia' => $scannedCount - $theoreticalCount,
                    'tipo' => $scannedCount > $theoreticalCount ? 'exceso' : 'faltante',
                ];
            }
        }

        // Buscar artículos escaneados que no están en teóricos
        foreach ($scanned as $mocaco => $scannedItem) {
            if (!isset($theoretical[$mocaco])) {
                $discrepancies[] = [
                    'mocaco' => $mocaco,
                    'unidades_teoricas' => 0,
                    'unidades_escaneadas' => $scannedItem->unidades_escaneadas,
                    'diferencia' => $scannedItem->unidades_escaneadas,
                    'tipo' => 'no_esperado',
                ];
            }
        }

        return [
            'container_entry_id' => $containerEntry->id,
            'n_camion' => $containerEntry->n_camion,
            'total_discrepancias' => count($discrepancies),
            'discrepancies' => $discrepancies,
            'summary' => [
                'total_teorico' => $theoretical->sum('unidades_teoricas'),
                'total_escaneado' => $scanned->sum('unidades_escaneadas'),
                'total_faltantes' => collect($discrepancies)->where('tipo', 'faltante')->sum('diferencia'),
                'total_excesos' => collect($discrepancies)->where('tipo', 'exceso')->sum('diferencia'),
                'total_no_esperados' => collect($discrepancies)->where('tipo', 'no_esperado')->sum('unidades_escaneadas'),
            ],
        ];
    }

    /**
     * Finalizar contenedor con validación de discrepancias
     * 
     * @param ContainerEntry $containerEntry
     * @param bool $force Si true, finaliza aunque haya discrepancias
     * @return array
     */
    public function finalizeContainer(ContainerEntry $containerEntry, bool $force = false): array
    {
        $report = $this->generateDiscrepancyReport($containerEntry);
        
        if (!$force && $report['total_discrepancias'] > 0) {
            return [
                'success' => false,
                'message' => 'No se puede finalizar: existen discrepancias entre el inventario teórico y escaneado.',
                'report' => $report,
            ];
        }

        try {
            DB::beginTransaction();
            
            $containerEntry->finalizar();
            
            DB::commit();
            
            return [
                'success' => true,
                'message' => 'Contenedor finalizado correctamente.',
                'report' => $report,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al finalizar contenedor: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Error al finalizar el contenedor: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Calcular tiempo de ciclo de procesamiento
     * 
     * @param ContainerEntry $containerEntry
     * @return array
     */
    public function calculateCycleTime(ContainerEntry $containerEntry): array
    {
        if (!$containerEntry->fecha_inicio_proceso || !$containerEntry->fecha_finalizacion) {
            return [
                'total_hours' => null,
                'total_minutes' => null,
                'formatted' => 'N/A',
            ];
        }

        $start = $containerEntry->fecha_inicio_proceso;
        $end = $containerEntry->fecha_finalizacion;
        $diff = $start->diffInMinutes($end);
        
        $hours = floor($diff / 60);
        $minutes = $diff % 60;

        return [
            'total_hours' => round($diff / 60, 2),
            'total_minutes' => $diff,
            'formatted' => "{$hours}h {$minutes}m",
            'start' => $start->format('d/m/Y H:i'),
            'end' => $end->format('d/m/Y H:i'),
        ];
    }
}

