<?php

namespace App\Services;

use App\Models\Inventory;
use Illuminate\Support\Facades\DB;

class BoxNumberService
{
    /**
     * Generar el siguiente número de caja para un contenedor
     * 
     * @param string|null $conteneur Número de contenedor
     * @param string $format Formato: 'container' (CONTENEDOR-SECUENCIA) o 'global' (CAJA-001)
     * @return string
     */
    public function generateNextBoxNumber(?string $conteneur = null, string $format = 'container'): string
    {
        if ($format === 'container' && $conteneur) {
            return $this->generateNextBoxNumberForContainer($conteneur);
        }
        
        return $this->generateNextGlobalBoxNumber();
    }

    /**
     * Generar número de caja basado en contenedor (ej: PT233I23-001, PT233I23-002)
     * Formato: CONTENEDOR-SECUENCIA
     */
    private function generateNextBoxNumberForContainer(string $conteneur): string
    {
        // Normalizar el nombre del contenedor
        $conteneurNormalized = strtoupper(trim($conteneur));
        
        // Buscar el último número de caja para este contenedor
        // Buscar patrones como: CONTENEDOR-SECUENCIA
        // Ejemplo: PT233I23-001, PT233I23-002
        $lastBox = Inventory::byContainer($conteneur)
            ->where('n_carton', 'like', $conteneurNormalized . '-%')
            ->select('n_carton')
            ->orderByRaw('CAST(SUBSTRING_INDEX(n_carton, "-", -1) AS UNSIGNED) DESC')
            ->first();

        if ($lastBox) {
            // Extraer el número de secuencia del último formato
            // Formato esperado: CONTENEDOR-SECUENCIA
            if (preg_match('/^' . preg_quote($conteneurNormalized, '/') . '-(\d+)$/i', $lastBox->n_carton, $matches)) {
                $lastSequence = (int) $matches[1];
                $nextSequence = $lastSequence + 1;
            } else {
                // Si el formato no coincide exactamente, intentar extraer solo la secuencia
                if (preg_match('/-(\d+)$/', $lastBox->n_carton, $seqMatches)) {
                    $nextSequence = (int) $seqMatches[1] + 1;
                } else {
                    $nextSequence = 1;
                }
            }
        } else {
            // Si no hay cajas previas con este formato, buscar cualquier caja del contenedor
            // para obtener el número más alto
            $allBoxes = Inventory::byContainer($conteneur)
                ->select('n_carton')
                ->distinct()
                ->pluck('n_carton')
                ->toArray();
            
            $maxSequence = 0;
            foreach ($allBoxes as $box) {
                // Intentar extraer secuencia si sigue el formato CONTENEDOR-SECUENCIA
                $escapedConteneur = preg_quote($conteneurNormalized, '/');
                if (preg_match('/^' . $escapedConteneur . '-(\d+)$/i', $box, $matches)) {
                    $seq = (int) $matches[1];
                    if ($seq > $maxSequence) {
                        $maxSequence = $seq;
                    }
                }
            }
            
            $nextSequence = $maxSequence + 1;
        }

        // Formato: CONTENEDOR-SECUENCIA (ej: PT233I23-001)
        return sprintf('%s-%03d', $conteneurNormalized, $nextSequence);
    }

    /**
     * Generar número de caja global (ej: CAJA-001, CAJA-002)
     */
    private function generateNextGlobalBoxNumber(): string
    {
        $prefix = 'CAJA';
        
        // Buscar el último número de caja global
        $lastBox = Inventory::where('n_carton', 'like', $prefix . '-%')
            ->select('n_carton')
            ->orderByRaw('CAST(SUBSTRING_INDEX(n_carton, "-", -1) AS UNSIGNED) DESC')
            ->first();

        if ($lastBox) {
            $parts = explode('-', $lastBox->n_carton);
            $lastNumber = (int) end($parts);
            $nextNumber = $lastNumber + 1;
        } else {
            $nextNumber = 1;
        }

        return sprintf('%s-%03d', $prefix, $nextNumber);
    }

    /**
     * Obtener el último número de caja usado para un contenedor
     */
    public function getLastBoxNumberForContainer(?string $conteneur = null): ?string
    {
        if (!$conteneur) {
            return null;
        }

        $conteneurNormalized = strtoupper(trim($conteneur));
        
        return Inventory::byContainer($conteneur)
            ->where('n_carton', 'like', $conteneurNormalized . '-%')
            ->select('n_carton')
            ->orderByRaw('CAST(SUBSTRING_INDEX(n_carton, "-", -1) AS UNSIGNED) DESC')
            ->value('n_carton');
    }

    /**
     * Validar si un número de caja ya existe
     */
    public function boxNumberExists(string $boxNumber): bool
    {
        return Inventory::byBox($boxNumber)->exists();
    }
}

