<?php

namespace App\Services;

use App\Models\Inventory;
use Illuminate\Support\Facades\DB;
use Dompdf\Dompdf;
use Dompdf\Options;
use Picqer\Barcode\BarcodeGeneratorPNG;

class BoxLabelService
{
    /**
     * Generar contenido ZPL para etiqueta de caja
     * Agrupa por SECTION y FAMILIA, suma unidades
     */
    public function generateBoxLabelZpl(string $nCarton, ?string $conteneur = null): string
    {
        $boxContent = $this->getBoxContentGrouped($nCarton, $conteneur);

        if ($boxContent['items']->isEmpty()) {
            throw new \Exception("No se encontró contenido para la caja {$nCarton}");
        }

        // Obtener mocaco dominante si existe
        $dominantMocaco = $this->getDominantMocaco($nCarton, $conteneur);

        $zpl = '^XA';
        $zpl .= '^FO20,20^ADN,36,20^FDCAJA: ' . strtoupper($nCarton) . '^FS';

        if ($conteneur) {
            $zpl .= '^FO20,60^ADN,24,12^FDCONTENEDOR: ' . strtoupper($conteneur) . '^FS';
        }

        $yPos = 100;

        // Mostrar mocaco dominante si existe (>50%)
        if ($dominantMocaco) {
            $zpl .= '^FO20,' . $yPos . '^ADN,28,14^FDARTICULO PRINCIPAL:^FS';
            $yPos += 35;
            $zpl .= '^FO20,' . $yPos . '^ADN,32,16^FD' . $dominantMocaco['count'] . ' x ' . $this->sanitizeForZpl($dominantMocaco['mocaco']) . '^FS';
            $yPos += 40;

            $rest = $dominantMocaco['total'] - $dominantMocaco['count'];
            if ($rest > 0) {
                $zpl .= '^FO20,' . $yPos . '^ADN,28,14^FD+ ' . $rest . ' x VARIOS^FS';
                $yPos += 35;
            }
        }

        $zpl .= '^FO20,' . $yPos . '^GB750,2,2^FS'; // Línea separadora
        $yPos += 15;

        // Resumen agrupado por FAMILIA/DETALLE USUARIO
        $zpl .= '^FO20,' . $yPos . '^ADN,24,12^FDRESUMEN POR FAMILIA/DETALLE USUARIO:^FS';
        $yPos += 30;

        foreach ($boxContent['items'] as $item) {
            if ($yPos > 800) { // Evitar desbordamiento
                $zpl .= '^FO20,' . $yPos . '^ADN,18,10^FD... más items ...^FS';
                break;
            }

            $userFamilyLine = 'FAMILIA USUARIO: ' . strtoupper($item->famillie_usuario ?? 'N/A');
            $detailLine = 'DETALLE USUARIO: ' . strtoupper($item->detail_usuario ?? 'N/A');
            $locationLine = 'UBICACION: ' . strtoupper($item->ubicacion ?? 'N/A');
            $seasonLine = 'SEASON: ' . ($item->season_int ?? 'N/A') . ' | UNIDADES: ' . $item->total_unidades;

            foreach ([$userFamilyLine, $detailLine, $locationLine, $seasonLine] as $line) {
                $zpl .= '^FO20,' . $yPos . '^ADN,18,10^FD' . $this->sanitizeForZpl($line) . '^FS';
                $yPos += 22;
            }

            $yPos += 6; // Espacio adicional entre bloques
        }

        $zpl .= '^FO20,' . $yPos . '^GB750,2,2^FS';
        $yPos += 15;

        // Total de unidades
        $zpl .= '^FO20,' . $yPos . '^ADN,30,15^FDTOTAL UNIDADES: ' . $boxContent['total_unidades'] . '^FS';

        // Código de barras de la caja
        $yPos += 50;
        $zpl .= '^FO20,' . $yPos . '^BY3^BCN,80,Y,N,N^FD' . $nCarton . '^FS';

        $zpl .= '^XZ';

        return $zpl;
    }

    /**
     * Generar PDF para etiqueta de caja (alternativa a ZPL)
     */
    public function generateBoxLabelPdf(string $nCarton, ?string $conteneur = null): string
    {
        $boxContent = $this->getBoxContentGrouped($nCarton, $conteneur);

        if ($boxContent['items']->isEmpty()) {
            throw new \Exception("No se encontró contenido para la caja {$nCarton}");
        }

        return $this->generateLabelHtml($nCarton, $conteneur, $boxContent);
    }

    /**
     * Generar PDF "CONTENU DU COLIS" con formato específico
     * Ahora reutiliza el mismo formato que la etiqueta de caja
     */
    public function generateContenuDuColisPdf(string $nCarton, ?string $conteneur = null, ?string $observations = null): \DomPDF\Dompdf
    {
        // Usamos getBoxContentGrouped para mantener el mismo formato de resumen
        $boxContent = $this->getBoxContentGrouped($nCarton, $conteneur);

        if ($boxContent['items']->isEmpty()) {
            throw new \Exception("No se encontró contenido para la caja {$nCarton}");
        }

        $html = $this->generateLabelHtml($nCarton, $conteneur, $boxContent, $observations);

        // Configurar DomPDF
        $options = new Options();
        $options->set('isHtml5ParserEnabled', true);
        $options->set('isRemoteEnabled', true);
        $options->set('defaultFont', 'Arial');

        $dompdf = new Dompdf($options);
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        return $dompdf;
    }

    /**
     * Generar HTML compartido para etiquetas y contenido de colis
     */
    private function generateLabelHtml(string $nCarton, ?string $conteneur, array $boxContent, ?string $observations = null): string
    {
        // Obtener la primera letra de la temporada (season_int) del primer ítem
        $firstItem = $boxContent['items']->first();
        $seasonLetter = $firstItem && $firstItem->season_int ? mb_strtoupper(mb_substr($firstItem->season_int, 0, 1)) : '';

        // Obtener mocaco dominante si existe
        $dominantMocaco = $this->getDominantMocaco($nCarton, $conteneur);

        // Generar HTML para PDF
        $html = '<!DOCTYPE html><html><head><meta charset="UTF-8">';
        $html .= '<style>
            body { font-family: Arial, sans-serif; padding: 30px; width: 90%; position: relative; }
            .header { font-size: 68px; font-weight: bold; margin-bottom: 20px; }
            .subheader { font-size: 52px; margin-bottom: 10px; }
            .dominant-mocaco {
                font-size: 56px;
                font-weight: bold;
                color: #2C3E50;
                background-color: #F8F9FA;
                padding: 15px;
                border: 3px solid #3498DB;
                border-radius: 8px;
                margin: 20px 0;
                text-align: center;
            }
            .dominant-mocaco .label {
                font-size: 36px;
                color: #7F8C8D;
                display: block;
                margin-bottom: 5px;
            }
            .dominant-mocaco .percentage {
                font-size: 32px;
                color: #27AE60;
                margin-left: 10px;
            }
            .observations { font-size: 36px; margin-bottom: 10px; font-style: italic; }
            .separator { border-top: 2px solid #000; margin: 20px 0; }
            .summary-title { font-size: 45px; font-weight: bold; margin: 20px 0; }
            .summary-item { font-size: 38px; margin: 15px 0; }
            .summary-item strong { display: block; font-size: 40px; }
            .summary-subitem { font-size: 45px; margin-left: 10px; color: #333; margin-top: 5px; }
            .total { font-size: 46px; font-weight: bold; margin-top: 28px; }
            .barcode { text-align: center; margin-top: 25px; }
            .barcode img { width: 80%; height: auto; }
            .season-big-letter {
                position: absolute;
                top: 0;
                right: 0;
                font-size: 180px;
                font-weight: bold;
                line-height: 1;
            }
        </style></head><body>';

        if ($seasonLetter) {
            $html .= '<div class="season-big-letter">' . $seasonLetter . '</div>';
        }

        $html .= '<div class="header">CAJA: ' . strtoupper($nCarton) . '</div>';

        if ($conteneur) {
            $html .= '<div class="subheader">CONTENEDOR: ' . strtoupper($conteneur) . '</div>';
        }

        // Mostrar mocaco dominante si existe (>50%)
        if ($dominantMocaco) {
            $html .= '<div class="dominant-mocaco">';
            $html .= '<span class="label">ARTÍCULO PRINCIPAL</span>';
            $html .= '<div>' . $dominantMocaco['count'] . ' x ' . htmlspecialchars($dominantMocaco['mocaco']) . '</div>';

            $rest = $dominantMocaco['total'] - $dominantMocaco['count'];
            if ($rest > 0) {
                $html .= '<div style="color: #7F8C8D; font-size: 0.8em; margin-top: 5px;">+ ' . $rest . ' x VARIOS</div>';
            }
            $html .= '</div>';
        }

        if ($observations) {
            $html .= '<div class="observations">Observations: ' . htmlspecialchars($observations) . '</div>';
        }

        $html .= '<div class="separator"></div>';

        $html .= '<div class="summary-title"></div>';

        foreach ($boxContent['items'] as $item) {
            $primaryLine = sprintf(
                ' %s %s',
                $item->seccion ?? 'N/A',
                $item->famillie_usuario ?? 'N/A'
            );
            $detailLine = 'Detalle usuario: ' . ($item->detail_usuario ?? 'N/A');
            // $locationLine = 'Ubicación: ' . ($item->ubicacion ?? 'N/A');
            $seasonLine = 'Season: ' . ($item->season_int ?? 'N/A') . ' | Unidades: ' . $item->total_unidades;

            $html .= '<div class="summary-item">'
                . '<strong>' . htmlspecialchars($primaryLine) . '</strong>'
                . '<div class="summary-subitem">' . htmlspecialchars($detailLine) . '</div>'
                // . '<div class="summary-subitem">' . htmlspecialchars($locationLine) . '</div>'
                . '<div class="summary-subitem">' . htmlspecialchars($seasonLine) . '</div>'
                . '</div>';
        }

        $html .= '<div class="separator"></div>';
        $html .= '<div class="total">TOTAL UNIDADES: ' . $boxContent['total_unidades'] . '</div>';
        $html .= '<div class="barcode"><img src="data:image/png;base64,' . $this->generateBarcodeImage($nCarton) . '" alt="' . $nCarton . '"></div>';
        $html .= '</body></html>';

        return $html;
    }

    /**
     * Calcular el mocaco dominante en una caja (>50%)
     * Retorna el mocaco si representa más del 50% del contenido, null en caso contrario
     */
    public function getDominantMocaco(string $nCarton, ?string $conteneur = null): ?array
    {
        $query = Inventory::byBox($nCarton);

        if ($conteneur) {
            $query->byContainer($conteneur);
        }

        // Obtener total de artículos en la caja
        $totalItems = $query->count();

        if ($totalItems === 0) {
            return null;
        }

        // Agrupar por mocaco y contar
        $mocacoStats = $query->select(
            'mocaco',
            DB::raw('COUNT(*) as count')
        )
            ->groupBy('mocaco')
            ->orderByDesc('count')
            ->first();

        if (!$mocacoStats) {
            return null;
        }

        // Calcular porcentaje
        $percentage = ($mocacoStats->count / $totalItems) * 100;

        // Solo retornar si es más del 50%
        if ($percentage > 50) {
            return [
                'mocaco' => $mocacoStats->mocaco,
                'count' => $mocacoStats->count,
                'total' => $totalItems,
                'percentage' => round($percentage, 1),
            ];
        }

        return null;
    }

    /**
     * Obtener contenido de caja agrupado por SECTION y FAMILIA
     */
    public function getBoxContentGrouped(string $nCarton, ?string $conteneur = null): array
    {
        $query = Inventory::byBox($nCarton);

        if ($conteneur) {
            $query->byContainer($conteneur);
        }

        // Get total count BEFORE grouping to get actual total units
        $totalUnidades = $query->count();

        $items = $query->select(
            'famillie_usuario',
            'detail_usuario',
            'ubicacion',
            'season_int',
            'seccion',
            DB::raw('COUNT(*) as total_unidades')
        )
            ->groupBy('famillie_usuario', 'detail_usuario', 'ubicacion', 'season_int', 'seccion')
            ->orderBy('famillie_usuario')
            ->orderBy('detail_usuario')
            ->orderBy('ubicacion')
            ->orderBy('season_int')
            ->orderBy('seccion')
            ->get();

        return [
            'items' => $items,
            'total_unidades' => $totalUnidades,
            'n_carton' => $nCarton,
            'conteneur' => $conteneur,
        ];
    }

    /**
     * Limpiar texto para ZPL
     */
    private function sanitizeForZpl(string $text): string
    {
        $text = strtoupper($text);
        return preg_replace('/[\^\~\|]/', '', $text) ?? '';
    }

    /**
     * Buscar caja por número para reimpresión
     */
    public function findBoxForReprint(string $nCarton): ?array
    {
        $inventory = Inventory::byBox($nCarton)->first();

        if (!$inventory) {
            return null;
        }

        return [
            'n_carton' => $inventory->n_carton,
            'conteneur' => $inventory->conteneur,
            'total_items' => Inventory::byBox($nCarton)->count(),
        ];
    }

    /**
     * Generar imagen de código de barras en base64 (para PDF)
     */
    private function generateBarcodeImage(string $code): string
    {
        try {
            $generator = new BarcodeGeneratorPNG();
            $barcode = $generator->getBarcode($code, $generator::TYPE_CODE_128, 3, 80);
            return base64_encode($barcode);
        } catch (\Exception $e) {
            // Si falla, retornar string vacío
            return '';
        }
    }

    /**
     * Obtener contenido detallado de caja agrupado por descripción completa
     */
    public function getBoxContentDetailed(string $nCarton, ?string $conteneur = null): array
    {
        $query = Inventory::byBox($nCarton);

        if ($conteneur) {
            $query->byContainer($conteneur);
        }

        // Obtener todos los items primero
        $allItems = $query->get();

        // Agrupar en PHP por la descripción generada (cadena + familia + campaña + sección)
        $grouped = [];

        foreach ($allItems as $item) {
            // Generar la clave de agrupación basada en la descripción
            $seccion = trim($item->seccion ?? '');
            $cadena = trim($item->cadena ?? '');
            $familia = trim($item->familia_articulo_description ?? '');
            $campana = trim($item->campana ?? '');

            // Clave de agrupación: sección + cadena + familia + campaña
            $key = strtoupper($seccion . '|' . $cadena . '|' . $familia . '|' . $campana);

            if (!isset($grouped[$key])) {
                $grouped[$key] = (object)[
                    'mocaco' => $item->mocaco ?? '',
                    'seccion' => $seccion,
                    'cadena' => $cadena,
                    'familia_articulo_description' => $familia,
                    'campana' => $campana,
                    'quantite' => 0,
                ];
            }

            $grouped[$key]->quantite++;
        }

        // Convertir a colección y ordenar
        $items = collect($grouped)->values()->sortBy(function ($item) {
            return $item->seccion . '|' . $item->mocaco;
        })->values();

        $totalUnidades = $allItems->count();

        return [
            'items' => $items,
            'total_unidades' => $totalUnidades,
            'n_carton' => $nCarton,
            'conteneur' => $conteneur,
        ];
    }

    /**
     * Generar descripción del artículo para el PDF
     */
    private function generateArticleDescription($item): string
    {
        $parts = [];

        if ($item->cadena) {
            $parts[] = strtoupper($item->cadena);
        }

        if ($item->familia_articulo_description) {
            $parts[] = strtoupper($item->familia_articulo_description);
        }

        if ($item->campana) {
            $parts[] = $item->campana;
        }

        if ($item->seccion) {
            $parts[] = $item->seccion;
        }

        return implode(' ', $parts) ?: 'ARTICULO';
    }

    /**
     * Generar código de barras en base64
     */
    private function generateBarcodeBase64(string $code): string
    {
        try {
            $generator = new BarcodeGeneratorPNG();
            // Tamaño aumentado significativamente: factor de escala 4 y altura 100 para mejor visibilidad
            $barcode = $generator->getBarcode($code, $generator::TYPE_CODE_128, 4, 100);
            return base64_encode($barcode);
        } catch (\Exception $e) {
            // Si falla, retornar string vacío
            return '';
        }
    }
}
