<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Casts\Attribute;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'operario_id',
        'client_id',
        'is_active',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Get the client associated with this user (for client portal).
     */
    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    /**
     * Get the orders created by this user.
     */
    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    /**
     * Get the audit logs for this user.
     */
    public function auditLogs(): HasMany
    {
        return $this->hasMany(AuditLog::class);
    }

    /**
     * Contenedores asignados (relación muchos a muchos)
     */
    public function containerEntries(): BelongsToMany
    {
        return $this->belongsToMany(ContainerEntry::class, 'container_entry_user')
            ->withPivot(['role', 'assigned_at'])
            ->withTimestamps();
    }

    /**
     * Obtener el contenedor asignado actual (prioridad: tabla pivote, luego usuario_asignado_id)
     * Devuelve el primer contenedor donde el usuario esté asignado
     */
    public function assignedContainer()
    {
        // Primero buscar en la tabla pivote (relación muchos a muchos)
        $container = $this->containerEntries()
            ->orderBy('container_entry_user.created_at', 'desc')
            ->first();

        // Si no se encuentra en la tabla pivote, buscar en el campo usuario_asignado_id
        if (!$container) {
            $container = ContainerEntry::where('usuario_asignado_id', $this->id)
                ->orderBy('created_at', 'desc')
                ->first();
        }

        return $container;
    }

    /**
     * Accessor para obtener el número de camión del contenedor asignado
     * Uso: Auth::user()->n_camion
     */
    protected function nCamion(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->assignedContainer()?->n_camion
        );
    }

    /**
     * Scope for active users.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function isSuperAdmin()
    {
        return $this->hasRole('admin');
    }
}
