<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class TheoreticalArticle extends Model
{
    use HasFactory;

    protected $fillable = [
        'container_entry_id',
        'mocaco',
        'cantidad_unidades',
        'season_int',
        'categoria_seleccionada',
        'famillie_usuario',
        'detail_usuario',
        'seccion',
        'familia_articulo_description',
        'cadena',
        'mercado_origen_articulo',
        'precio_pvp_maximo_temporada',
        'partida_arancelaria',
        'composition',
        'campana',
        'peso_unitario',
        'grupo_arancelario',
        'notes',
    ];

    protected $casts = [
        'precio_pvp_maximo_temporada' => 'decimal:2',
        'peso_unitario' => 'decimal:3',
        'cantidad_unidades' => 'integer',
    ];

    /**
     * Get the container entry that owns this theoretical article.
     */
    public function containerEntry(): BelongsTo
    {
        return $this->belongsTo(ContainerEntry::class);
    }

    /**
     * Get the article that matches this theoretical article.
     */
    public function article(): BelongsTo
    {
        return $this->belongsTo(Article::class, 'mocaco', 'mocaco');
    }

    /**
     * Scope for filtering by container entry.
     */
    public function scopeByContainerEntry($query, int $containerEntryId)
    {
        return $query->where('container_entry_id', $containerEntryId);
    }

    /**
     * Scope for filtering by mocaco.
     */
    public function scopeByMocaco($query, string $mocaco)
    {
        $trimmedMocaco = trim($mocaco);
        return $query->whereRaw('LOWER(TRIM(mocaco)) = LOWER(?)', [$trimmedMocaco]);
    }

    /**
     * Get total weight for this theoretical article (peso_unitario * cantidad_unidades).
     */
    public function getPesoTotalAttribute(): float
    {
        return ($this->peso_unitario ?? 0) * $this->cantidad_unidades;
    }

    /**
     * Get total price for this theoretical article (precio * cantidad_unidades).
     */
    public function getPrecioTotalAttribute(): float
    {
        return ($this->precio_pvp_maximo_temporada ?? 0) * $this->cantidad_unidades;
    }
}
