<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Brand extends Model
{
    protected $fillable = [
        'name',
        'aliases',
        'description',
        'is_active',
    ];

    protected $casts = [
        'aliases' => 'array',
        'is_active' => 'boolean',
    ];

    /**
     * Normalize a brand name.
     * Checks against official name and aliases.
     * Returns the official name if found, otherwise returns the input.
     *
     * @param string|null $inputName
     * @return string|null
     */
    public static function normalize(?string $inputName, bool $strict = false): ?string
    {
        if (empty($inputName)) {
            return $inputName;
        }

        $inputNameTrimmed = trim($inputName);

        // 1. Check strict match with official name
        $brand = self::where('name', $inputNameTrimmed)->first();
        if ($brand) {
            return $brand->name;
        }

        // 2. Check aliases (using JSON search or fetching all and filtering)
        // Since dataset is likely small, fetching active brands is efficient enough.
        $brands = self::where('is_active', true)->get();

        foreach ($brands as $brand) {
            if ($brand->aliases && is_array($brand->aliases)) {
                foreach ($brand->aliases as $alias) {
                    if (strcasecmp($alias, $inputNameTrimmed) === 0) {
                        return $brand->name;
                    }
                }
            }

            // Also case-insensitive check against name
            if (strcasecmp($brand->name, $inputNameTrimmed) === 0) {
                return $brand->name;
            }
        }

        if ($strict) {
            throw new \Exception("La marca '{$inputNameTrimmed}' no se encuentra en el Maestro de Marcas.");
        }

        return $inputNameTrimmed;
    }

    /**
     * Check if a brand name exists (as official name or alias).
     * Returns true if found, false otherwise.
     */
    public static function exists(?string $inputName): bool
    {
        if (empty($inputName)) {
            return true; // Empty brands are allowed
        }

        $inputNameTrimmed = trim($inputName);

        // Check strict match with official name
        if (self::where('name', $inputNameTrimmed)->exists()) {
            return true;
        }

        // Check aliases
        $brands = self::where('is_active', true)->get();
        foreach ($brands as $brand) {
            if ($brand->aliases && is_array($brand->aliases)) {
                foreach ($brand->aliases as $alias) {
                    if (strcasecmp($alias, $inputNameTrimmed) === 0) {
                        return true;
                    }
                }
            }
            // Case-insensitive check against name
            if (strcasecmp($brand->name, $inputNameTrimmed) === 0) {
                return true;
            }
        }

        return false;
    }
}
