<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BoxReservationItem extends Model
{
    protected $fillable = [
        'reservation_id',
        'inventory_id',
        'box_number',
        'quantity',
        'status',
        'scanned_at',
        'notes',
    ];

    protected $casts = [
        'scanned_at' => 'datetime',
    ];

    /**
     * Get the reservation that owns the item
     */
    public function reservation(): BelongsTo
    {
        return $this->belongsTo(BoxReservation::class, 'reservation_id');
    }

    /**
     * Get the inventory item
     */
    public function inventory(): BelongsTo
    {
        return $this->belongsTo(Inventory::class);
    }

    /**
     * Check if item is reserved
     */
    public function isReserved(): bool
    {
        return $this->status === 'reserved';
    }

    /**
     * Check if item is confirmed
     */
    public function isConfirmed(): bool
    {
        return $this->status === 'confirmed';
    }

    /**
     * Check if item is cancelled
     */
    public function isCancelled(): bool
    {
        return $this->status === 'cancelled';
    }

    /**
     * Mark item as confirmed
     */
    public function confirm(): bool
    {
        return $this->update([
            'status' => 'confirmed',
            'scanned_at' => now(),
        ]);
    }

    /**
     * Mark item as cancelled
     */
    public function cancel(): bool
    {
        return $this->update([
            'status' => 'cancelled',
        ]);
    }

    /**
     * Scope: Only reserved items
     */
    public function scopeReserved($query)
    {
        return $query->where('status', 'reserved');
    }

    /**
     * Scope: Only confirmed items
     */
    public function scopeConfirmed($query)
    {
        return $query->where('status', 'confirmed');
    }
}
