<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class BoxReservation extends Model
{
    protected $fillable = [
        'client_id',
        'reservation_code',
        'status',
        'reserved_by_user_id',
        'confirmed_by_user_id',
        'notes',
        'reserved_at',
        'confirmed_at',
    ];

    protected $casts = [
        'reserved_at' => 'datetime',
        'confirmed_at' => 'datetime',
    ];

    /**
     * Get the client that owns the reservation
     */
    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    /**
     * Get the user who created the reservation
     */
    public function reservedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reserved_by_user_id');
    }

    /**
     * Get the user who confirmed the reservation
     */
    public function confirmedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'confirmed_by_user_id');
    }

    /**
     * Get the reservation items
     */
    public function items(): HasMany
    {
        return $this->hasMany(BoxReservationItem::class, 'reservation_id');
    }

    /**
     * Get confirmed items only
     */
    public function confirmedItems(): HasMany
    {
        return $this->hasMany(BoxReservationItem::class, 'reservation_id')
            ->where('status', 'confirmed');
    }

    /**
     * Get pending items (reserved but not confirmed)
     */
    public function pendingItems(): HasMany
    {
        return $this->hasMany(BoxReservationItem::class, 'reservation_id')
            ->where('status', 'reserved');
    }

    /**
     * Get total boxes count
     */
    public function getTotalBoxesAttribute(): int
    {
        return $this->items()->count();
    }

    /**
     * Get confirmed boxes count
     */
    public function getConfirmedBoxesAttribute(): int
    {
        return $this->confirmedItems()->count();
    }

    /**
     * Get pending boxes count
     */
    public function getPendingBoxesAttribute(): int
    {
        return $this->pendingItems()->count();
    }

    /**
     * Get completion percentage
     */
    public function getCompletionPercentageAttribute(): float
    {
        if ($this->total_boxes == 0) {
            return 0;
        }
        return round(($this->confirmed_boxes / $this->total_boxes) * 100, 2);
    }

    /**
     * Check if reservation is pending
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if reservation is in progress
     */
    public function isInProgress(): bool
    {
        return $this->status === 'in_progress';
    }

    /**
     * Check if reservation is completed
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if reservation is cancelled
     */
    public function isCancelled(): bool
    {
        return $this->status === 'cancelled';
    }

    /**
     * Generate unique reservation code
     */
    public static function generateReservationCode(): string
    {
        $date = now()->format('Ymd');
        $count = self::whereDate('created_at', now())->count() + 1;
        return 'RES-' . $date . '-' . str_pad($count, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Scope: Only pending reservations
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope: Only in progress reservations
     */
    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    /**
     * Scope: Active reservations (pending or in progress)
     */
    public function scopeActive($query)
    {
        return $query->whereIn('status', ['pending', 'in_progress']);
    }
}
