<?php

namespace App\Jobs;

use App\Models\ImportProgress;
use App\Models\ContainerEntry;
use App\Imports\TheoreticalArticleImport;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;

class ProcessTheoreticalImportJob implements ShouldQueue
{
    use Queueable, InteractsWithQueue, SerializesModels;

    public $timeout = 7200; // 2 hours per chunk
    public $tries = 1;      // No retries - first error is the real one
    public $maxExceptions = 1;

    protected $importProgressId;
    protected $filePath;

    /**
     * Create a new job instance.
     */
    public function __construct(int $importProgressId, string $filePath)
    {
        $this->importProgressId = $importProgressId;
        $this->filePath = $filePath;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $importProgress = ImportProgress::find($this->importProgressId);

        if (!$importProgress) {
            Log::error("ImportProgress not found: {$this->importProgressId}");
            return;
        }

        try {
            // Update status to processing
            $importProgress->update([
                'status' => 'processing',
                'started_at' => now(),
            ]);

            // Check if file exists
            if (!Storage::disk('local')->exists($this->filePath)) {
                throw new \Exception("File not found: {$this->filePath}");
            }

            // Get the full path
            $fullPath = Storage::disk('local')->path($this->filePath);

            // ========================================
            // STEP 1: Pre-validate all brands in file
            // ========================================

            $invalidBrands = $this->validateBrandsInFile($fullPath);

            if (!empty($invalidBrands)) {
                $brandList = implode(', ', array_unique($invalidBrands));
                throw new \Exception("Las siguientes marcas no están registradas en el Maestro de Marcas: {$brandList}. Por favor, agrégalas antes de importar.");
            }

            // ========================================
            // STEP 2: Proceed with actual import
            // ========================================
            $importProgress->update(['status' => 'processing']);

            // Create import instance
            $import = new TheoreticalArticleImport($importProgress->container_entry_id);

            // Set progress callback to update database
            $import->setProgressCallback(function ($processedRows, $totalRows) use ($importProgress) {
                $importProgress->update([
                    'processed_rows' => $processedRows,
                    'total_rows' => $totalRows,
                ]);
            });

            // Import the file
            Excel::import($import, $fullPath);

            // Get final stats
            $stats = $import->getStats();

            // Update progress as completed
            $importProgress->update([
                'status' => 'completed',
                'completed_at' => now(),
                'processed_rows' => $stats['imported'] + $stats['skipped'],
                'imported_rows' => $stats['imported'],
                'skipped_rows' => $stats['skipped'],
                'total_rows' => $stats['imported'] + $stats['skipped'],
            ]);

            // Clean up the temporary file
            Storage::disk('local')->delete($this->filePath);

            Log::info("Import completed successfully", [
                'import_progress_id' => $this->importProgressId,
                'imported' => $stats['imported'],
                'skipped' => $stats['skipped'],
            ]);
        } catch (\Exception $e) {
            // Update progress as failed
            $importProgress->update([
                'status' => 'failed',
                'completed_at' => now(),
                'error_message' => $e->getMessage(),
            ]);

            // Clean up the temporary file
            if (Storage::disk('local')->exists($this->filePath)) {
                Storage::disk('local')->delete($this->filePath);
            }

            Log::error("Import failed", [
                'import_progress_id' => $this->importProgressId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            // Re-throw to mark job as failed
            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        $importProgress = ImportProgress::find($this->importProgressId);

        if ($importProgress) {
            $importProgress->update([
                'status' => 'failed',
                'completed_at' => now(),
                'error_message' => $exception->getMessage(),
            ]);
        }

        // Clean up the temporary file
        if (Storage::disk('local')->exists($this->filePath)) {
            Storage::disk('local')->delete($this->filePath);
        }

        Log::error("Import job failed permanently", [
            'import_progress_id' => $this->importProgressId,
            'error' => $exception->getMessage(),
        ]);
    }

    /**
     * Validate all brands in the file before importing.
     * Returns array of invalid brand names.
     */
    protected function validateBrandsInFile(string $fullPath): array
    {
        $invalidBrands = [];

        // Use PhpSpreadsheet to read the file directly for validation
        $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReaderForFile($fullPath);
        $reader->setReadDataOnly(true);
        $spreadsheet = $reader->load($fullPath);
        $worksheet = $spreadsheet->getActiveSheet();

        // Find the 'cadena' column index
        $headerRow = $worksheet->getRowIterator(1, 1)->current();
        $cadenaColumnIndex = null;

        foreach ($headerRow->getCellIterator() as $cell) {
            $header = strtolower(trim($cell->getValue() ?? ''));
            if ($header === 'cadena') {
                $cadenaColumnIndex = $cell->getColumn();
                break;
            }
        }

        if ($cadenaColumnIndex === null) {
            // No 'cadena' column found, no brand validation needed
            return [];
        }

        // Iterate through all data rows and collect unique brands
        $uniqueBrands = [];
        $highestRow = $worksheet->getHighestRow();

        for ($row = 2; $row <= $highestRow; $row++) {
            $brandValue = $worksheet->getCell($cadenaColumnIndex . $row)->getValue();
            if (!empty($brandValue)) {
                $brandTrimmed = trim($brandValue);
                if (!isset($uniqueBrands[$brandTrimmed])) {
                    $uniqueBrands[$brandTrimmed] = true;
                }
            }
        }

        // Check each unique brand
        foreach (array_keys($uniqueBrands) as $brandName) {
            if (!\App\Models\Brand::exists($brandName)) {
                $invalidBrands[] = $brandName;
            }
        }

        return $invalidBrands;
    }
}
