<?php

namespace App\Imports;

use App\Models\Client;
use App\Models\Inventory;
use App\Models\Order;
use App\Models\OrderItem;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class OrderReservationsImport implements ToCollection, WithHeadingRow
{
    protected $userId;

    protected array $summary = [
        'processed_rows' => 0,
        'reserved_boxes' => 0,
        'reserved_units' => 0,
        'orders_created' => 0,
        'orders_used' => 0,
        'skipped_rows' => 0,
        'errors' => [],
    ];

    /**
     * Cache para pedidos creados o cargados durante el import.
     *
     * @var array<int,\App\Models\Order>
     */
    protected array $ordersCache = [];

    public function __construct(?int $userId = null)
    {
        $this->userId = $userId;
    }

    /**
     * @param Collection<int,array<string,mixed>> $rows
     */
    public function collection(Collection $rows)
    {
        foreach ($rows as $index => $row) {
            $rowNumber = $index + 2; // Considerando fila de encabezados
            $this->summary['processed_rows']++;

            $clientIdentifier = $this->stringValue(
                Arr::get($row, 'client_id', Arr::get($row, 'cliente'))
            );
            $boxNumber = $this->stringValue(
                Arr::get($row, 'n_carton', Arr::get($row, 'caja'))
            );
            $orderIdRaw = $this->stringValue(Arr::get($row, 'pedido_id'));
            $observation = $this->stringValue(Arr::get($row, 'observacion'));

            if (empty($clientIdentifier) && empty($orderIdRaw)) {
                $this->addError($rowNumber, 'Debe indicar el cliente o el ID de pedido.');
                $this->summary['skipped_rows']++;
                continue;
            }

            if (empty($boxNumber)) {
                $this->addError($rowNumber, 'El número de caja (n_carton) es obligatorio.');
                $this->summary['skipped_rows']++;
                continue;
            }

            try {
                DB::beginTransaction();

                $order = $this->resolveOrder($clientIdentifier, $orderIdRaw, $rowNumber);
                if (!$order) {
                    DB::rollBack();
                    $this->summary['skipped_rows']++;
                    continue;
                }

                $reservedUnits = $this->reserveBoxForOrder($order, $boxNumber, $observation, $rowNumber);

                if ($reservedUnits > 0) {
                    $this->summary['reserved_boxes']++;
                    $this->summary['reserved_units'] += $reservedUnits;
                } else {
                    $this->summary['skipped_rows']++;
                }

                DB::commit();
            } catch (\Throwable $e) {
                DB::rollBack();
                $this->addError($rowNumber, $e->getMessage());
                $this->summary['skipped_rows']++;
            }
        }
    }

    public function getSummary(): array
    {
        return $this->summary;
    }

    protected function resolveOrder(string $clientIdentifier = null, ?string $orderIdRaw = null, int $rowNumber = 0): ?Order
    {
        if (!empty($orderIdRaw)) {
            $orderId = (int) $orderIdRaw;
            if (!$orderId) {
                $this->addError($rowNumber, 'El ID de pedido proporcionado no es válido.');
                return null;
            }

            if (!isset($this->ordersCache[$orderId])) {
                $order = Order::with('client')->find($orderId);
                if (!$order) {
                    $this->addError($rowNumber, "El pedido #{$orderId} no existe.");
                    return null;
                }

                if (!in_array($order->status, ['draft', 'reserved'], true)) {
                    $this->addError($rowNumber, "El pedido #{$orderId} no permite agregar reservas (estado actual: {$order->status}).");
                    return null;
                }

                $this->ordersCache[$orderId] = $order;
                $this->summary['orders_used']++;
            }

            return $this->ordersCache[$orderId];
        }

        if (empty($clientIdentifier)) {
            $this->addError($rowNumber, 'No se pudo determinar el cliente para esta fila.');
            return null;
        }

        $client = $this->findClient($clientIdentifier);

        if (!$client) {
            $this->addError($rowNumber, "Cliente no encontrado: {$clientIdentifier}");
            return null;
        }

        $cacheKey = 'client_' . $client->id;
        if (!isset($this->ordersCache[$cacheKey])) {
            $order = Order::create([
                'client_id' => $client->id,
                'user_id' => $this->userId,
                'status' => 'reserved',
                'order_date' => now(),
                'notes' => 'Pedido creado por importación de reservas el ' . now()->format('d/m/Y H:i'),
            ]);

            $this->ordersCache[$cacheKey] = $order;
            $this->summary['orders_created']++;
        }

        return $this->ordersCache[$cacheKey];
    }

    protected function reserveBoxForOrder(Order $order, string $boxNumber, ?string $observation, int $rowNumber): int
    {
        $items = Inventory::where('n_carton', $boxNumber)
            ->lockForUpdate()
            ->get();

        if ($items->isEmpty()) {
            $this->addError($rowNumber, "La caja {$boxNumber} no existe en el inventario.");
            return 0;
        }

        $conflict = $items->first(function ($item) use ($order) {
            if (in_array($item->status, ['dispatched', 'despachado'], true)) {
                return true;
            }

            if ($item->status === 'reservado' && (int) $item->order_id !== (int) $order->id) {
                return true;
            }

            return false;
        });

        if ($conflict) {
            $this->addError($rowNumber, "La caja {$boxNumber} ya está reservada o despachada.");
            return 0;
        }

        $itemsToReserve = $items->filter(function ($item) {
            return $item->status !== 'reservado' || !$item->order_id;
        });

        if ($itemsToReserve->isEmpty()) {
            // Ya estaba reservada por el mismo pedido
            return 0;
        }

        foreach ($itemsToReserve as $inventory) {
            OrderItem::firstOrCreate(
                [
                    'order_id' => $order->id,
                    'inventory_id' => $inventory->id,
                ],
                [
                    'full_barcode' => $inventory->full_barcode,
                    'mocaco' => $inventory->mocaco,
                    'n_carton' => $inventory->n_carton,
                ]
            );

            $inventory->status = 'reservado';
            $inventory->order_id = $order->id;

            if (!empty($observation)) {
                $note = 'Reserva importada: ' . $observation;
                $inventory->notes = $inventory->notes
                    ? Str::limit($inventory->notes . ' | ' . $note, 5000)
                    : $note;
            }

            $inventory->save();
        }

        return $itemsToReserve->count();
    }

    protected function findClient(string $identifier): ?Client
    {
        if (is_numeric($identifier)) {
            $client = Client::find((int) $identifier);
            if ($client) {
                return $client;
            }
        }

        return Client::whereRaw('LOWER(client_name) = ?', [mb_strtolower($identifier)])->first();
    }

    protected function addError(int $rowNumber, string $message): void
    {
        $this->summary['errors'][] = "Fila {$rowNumber}: {$message}";
    }

    protected function stringValue($value): string
    {
        if ($value === null) {
            return '';
        }

        return trim((string) $value);
    }
}

