<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Spatie\Permission\Models\Role;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Verificar permisos
        if (!Auth::user()->can('users.view')) {
            abort(403, 'No tienes permiso para ver usuarios.');
        }

        $query = User::query();

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('operario_id', 'like', "%{$search}%");
            });
        }

        if ($request->filled('role')) {
            $query->role($request->role);
        }

        if ($request->filled('is_active')) {
            $query->where('is_active', $request->is_active);
        }

        $perPage = $request->input('per_page', 20);
        $users = $query->with('roles')->orderBy('name')->paginate($perPage)->appends($request->except('page'));
        $roles = Role::orderBy('name')->get();

        return view('users.index', compact('users', 'roles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Verificar permisos
        if (!Auth::user()->can('users.create')) {
            abort(403, 'No tienes permiso para crear usuarios.');
        }

        $roles = Role::orderBy('name')->get();
        $clients = Client::orderBy('client_name')->get();
        return view('users.create', compact('roles', 'clients'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Verificar permisos
        if (!Auth::user()->can('users.create')) {
            abort(403, 'No tienes permiso para crear usuarios.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'operario_id' => 'nullable|string|max:50',
            'client_id' => 'nullable|exists:clients,id',
            'roles' => 'required|array',
            'roles.*' => 'exists:roles,name',
            'is_active' => 'boolean',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'operario_id' => $request->operario_id ?? null,
            'client_id' => $request->client_id ?? null,
            'is_active' => $request->is_active ?? true,
        ]);

        $user->assignRole($request->roles);

        return redirect()->route('users.index')
            ->with('success', 'Usuario creado exitosamente.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        $user->load('roles', 'permissions');
        return view('users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        // Verificar permisos
        if (!Auth::user()->can('users.update')) {
            abort(403, 'No tienes permiso para editar usuarios.');
        }

        $roles = Role::orderBy('name')->get();
        $clients = Client::orderBy('client_name')->get();
        $user->load('roles', 'client');
        return view('users.edit', compact('user', 'roles', 'clients'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        // Verificar permisos
        if (!Auth::user()->can('users.update')) {
            abort(403, 'No tienes permiso para actualizar usuarios.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('users')->ignore($user->id)],
            'password' => 'nullable|string|min:8|confirmed',
            'operario_id' => 'nullable|string|max:50',
            'client_id' => 'nullable|exists:clients,id',
            'roles' => 'required|array',
            'roles.*' => 'exists:roles,name',
            'is_active' => 'boolean',
        ]);

        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'operario_id' => $request->operario_id ?? null,
            'client_id' => $request->client_id ?? null,
            'is_active' => $request->is_active ?? true,
        ];

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $user->update($data);

        $user->syncRoles($request->roles);

        return redirect()->route('users.index')
            ->with('success', 'Usuario actualizado exitosamente.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Verificar permisos
        if (!Auth::user()->can('users.delete')) {
            abort(403, 'No tienes permiso para eliminar usuarios.');
        }

        if ($user->id === auth()->id()) {
            return redirect()->route('users.index')
                ->with('error', 'No puedes eliminar tu propio usuario.');
        }

        $user->delete();

        return redirect()->route('users.index')
            ->with('success', 'Usuario eliminado exitosamente.');
    }
}
