<?php

namespace App\Http\Controllers;

use App\Models\RelabelingProduct;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class TestRelabelingController extends Controller
{
    /**
     * Search for relabeling product by provider code.
     */
    public function searchProduct(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'proveedor_code' => 'required|string|max:50',
            ]);

            $product = RelabelingProduct::byProviderCode($request->proveedor_code)->first();

            if ($product) {
                return response()->json([
                    'success' => true,
                    'product' => $product,
                    'message' => 'Producto encontrado.',
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Producto no encontrado.',
            ], 404);
        } catch (\Exception $e) {
            \Log::error('Error en searchProduct: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error al buscar producto: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Save relabeling product.
     */
    public function saveProduct(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'proveedor_code' => 'required|string|max:50',
                'codebar_gnu' => 'required|string|max:50',
                'description' => 'required|string',
                'price_cfa' => 'nullable|numeric',
            ]);

            $product = RelabelingProduct::updateOrCreate(
                ['proveedor_code' => $request->proveedor_code],
                [
                    'codebar_gnu' => $request->codebar_gnu,
                    'description' => $request->description,
                    'price_cfa' => $request->price_cfa,
                ]
            );

            // Log the action
            AuditLog::logAction(
                'save_relabeling_product',
                $product,
                null, // No user for test
                [],
                $product->toArray(),
                "Producto de reetiquetado guardado: {$request->proveedor_code}"
            );

            return response()->json([
                'success' => true,
                'product' => $product,
                'message' => 'Producto guardado con éxito.',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error en saveProduct: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error al guardar el producto: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generate ZPL content for label printing.
     */
    public function generateZpl(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'barcode' => 'required|string|max:50',
                'my_code' => 'required|string|max:50',
                'description' => 'required|string',
                'price' => 'nullable|string',
            ]);

            $zpl = $this->createZplContent([
                'barcode' => $request->barcode,
                'my_code' => $request->my_code,
                'desc' => $request->description,
                'price' => $request->price ?? '0',
            ]);

            return response()->json([
                'success' => true,
                'zpl_content' => $zpl,
                'message' => 'Contenido ZPL generado.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error en generateZpl: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error al generar ZPL: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get all relabeling products.
     */
    public function getAllProducts(): JsonResponse
    {
        try {
            $products = RelabelingProduct::orderBy('proveedor_code')->get();

            return response()->json([
                'success' => true,
                'products' => $products,
            ]);
        } catch (\Exception $e) {
            \Log::error('Error en getAllProducts: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener productos: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete relabeling product.
     */
    public function deleteProduct(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'id' => 'required|exists:relabeling_products,id',
            ]);

            $product = RelabelingProduct::findOrFail($request->id);
            $productData = $product->toArray();
            
            $product->delete();

            // Log the action
            AuditLog::logAction(
                'delete_relabeling_product',
                $product,
                null, // No user for test
                $productData,
                [],
                "Producto de reetiquetado eliminado: {$productData['proveedor_code']}"
            );

            return response()->json([
                'success' => true,
                'message' => 'Producto eliminado correctamente.',
            ]);

        } catch (\Exception $e) {
            \Log::error('Error en deleteProduct: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el producto: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Create ZPL content for Zebra printer.
     */
    private function createZplContent(array $data): string
    {
        $zpl = '^XA';
        
        // Description
        $zpl .= '^FO20,20^ADN,36,20^FD' . $data['desc'] . '^FS';
        
        // Internal Code
        $zpl .= '^FO20,65^ADN,18,10^FDCÓDIGO INT: ' . $data['my_code'] . '^FS';
        
        // Price
        $zpl .= '^FO20,100^A0N,60,60^FDCFA ' . $data['price'] . '^FS';
        
        // Barcode
        $zpl .= '^FO20,170^BY3^BCN,100,Y,N,N^FD' . $data['barcode'] . '^FS';
        
        $zpl .= '^XZ';
        
        return $zpl;
    }
}