<?php

namespace App\Http\Controllers;

use App\Models\Article;
use App\Models\Inventory;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TestController extends Controller
{
    public function searchArticle(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'full_barcode' => 'required|string|max:50',
            ]);

            $fullBarcode = $request->input('full_barcode');
            $mocaco = Article::extractMocacoFromBarcode($fullBarcode);

            $articles = Article::where('mocaco', $mocaco)->get();

            return response()->json([
                'success' => true,
                'articles' => $articles,
                'message' => 'Búsqueda completada.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error en searchArticle: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error al buscar artículo: ' . $e->getMessage(),
                'articles' => []
            ], 500);
        }
    }

    public function processScan(Request $request): JsonResponse
    {
        $request->validate([
            'full_barcode' => 'required|string|max:50',
            'mocaco' => 'required|string|max:20',
            'n_carton' => 'required|string|max:50',
            'season_int' => 'required|string|max:20',
            'conteneur' => 'required|string|max:50',
            'categoria_seleccionada' => 'required|string|max:20',
            'famillie_usuario' => 'required|string|max:50',
            'detail_usuario' => 'nullable|string|max:100',
            'ubicacion' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            // Get article data
            $article = Article::where('mocaco', $request->mocaco)->first();
            if (!$article) {
                return response()->json([
                    'success' => false,
                    'message' => 'Artículo no encontrado en la base de datos.',
                ], 404);
            }

            // Create inventory record
            $inventory = Inventory::create([
                'full_barcode' => $request->full_barcode,
                'mocaco' => $request->mocaco,
                'n_carton' => $request->n_carton,
                'season_int' => $request->season_int,
                'conteneur' => $request->conteneur,
                'categoria_seleccionada' => $request->categoria_seleccionada,
                'famillie_usuario' => $request->famillie_usuario,
                'detail_usuario' => $request->detail_usuario,
                'ubicacion' => $request->ubicacion,
                'notes' => $request->notes,
                'n_id_operario' => 'TEST_USER', // Usuario de prueba
                'fecha_escaneo' => now(),
                // Copy article data
                'seccion' => $article->seccion,
                'familia_articulo_description' => $article->familia,
                'cadena' => $article->cadena,
                'mercado_origen_articulo' => $article->mercado_origen_articulo,
                'precio_pvp_maximo_temporada' => $article->precio_pvp_maximo_temporada,
                'partida_arancelaria' => $article->partida_arancelaria,
                'composition' => $article->composition,
                'campana' => $article->campana,
                'peso_unitario' => $article->peso_unitario,
                'grupo_arancelario' => $article->grupo_arancelario,
            ]);

            // Log the action
            AuditLog::logAction(
                'scan',
                $inventory,
                null, // No user for test
                [],
                $inventory->toArray(),
                "Artículo escaneado: {$request->full_barcode}"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Artículo escaneado y guardado correctamente.',
                'inventory' => $inventory,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error al procesar el escaneo: ' . $e->getMessage(),
            ], 500);
        }
    }
}