<?php

namespace App\Http\Controllers;

use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Solo admin puede ver roles
        if (!Auth::user()->hasRole('admin')) {
            abort(403, 'No tienes permiso para ver roles.');
        }

        $query = Role::query();

        // Apply search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }

        $roles = $query->with('permissions')->orderBy('name')->paginate(20);
        $allPermissions = Permission::orderBy('name')->get();

        return view('roles.index', compact('roles', 'allPermissions'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Solo admin puede crear roles
        if (!Auth::user()->hasRole('admin')) {
            abort(403, 'No tienes permiso para crear roles.');
        }

        $permissions = Permission::orderBy('name')->get();
        return view('roles.create', compact('permissions'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Solo admin puede crear roles
        if (!Auth::user()->hasRole('admin')) {
            abort(403, 'No tienes permiso para crear roles.');
        }

        $request->validate([
            'name' => 'required|string|max:255|unique:roles,name',
            'permissions' => 'required|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role = Role::create([
            'name' => $request->name,
            'guard_name' => 'web',
        ]);

        // Convertir IDs a objetos Permission
        $permissions = Permission::whereIn('id', $request->permissions)->get();
        $role->syncPermissions($permissions);

        return redirect()->route('roles.index')
            ->with('success', 'Rol creado exitosamente.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Role $role)
    {
        $role->load('permissions');
        $allPermissions = Permission::orderBy('name')->get();
        return view('roles.show', compact('role', 'allPermissions'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Role $role)
    {
        // Solo admin puede editar roles
        if (!Auth::user()->hasRole('admin')) {
            abort(403, 'No tienes permiso para editar roles.');
        }

        $permissions = Permission::orderBy('name')->get();
        $role->load('permissions');
        return view('roles.edit', compact('role', 'permissions'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Role $role)
    {
        // Solo admin puede actualizar roles
        if (!Auth::user()->hasRole('admin')) {
            abort(403, 'No tienes permiso para actualizar roles.');
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('roles')->ignore($role->id)],
            'permissions' => 'required|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role->update([
            'name' => $request->name,
        ]);

        // Convertir IDs a objetos Permission
        $permissions = Permission::whereIn('id', $request->permissions)->get();
        $role->syncPermissions($permissions);

        return redirect()->route('roles.index')
            ->with('success', 'Rol actualizado exitosamente.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Role $role)
    {
        // Check if role is system role (cannot be deleted)
        $systemRoles = ['admin', 'manager', 'operator', 'scanner', 'viewer'];
        
        if (in_array($role->name, $systemRoles)) {
            return redirect()->route('roles.index')
                ->with('error', 'No se puede eliminar un rol del sistema.');
        }

        $role->delete();

        return redirect()->route('roles.index')
            ->with('success', 'Rol eliminado exitosamente.');
    }
}

