<?php

namespace App\Http\Controllers;

use App\Models\RelabelingProduct;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class RelabelingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Mostrar vista de escaneo/impresión principal
        return view('relabeling.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('relabeling.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'proveedor_code' => 'required|string|max:50|unique:relabeling_products,proveedor_code',
            'codebar_gnu' => 'required|string|max:50',
            'description' => 'required|string',
            'price_cfa' => 'nullable|numeric',
        ]);

        $product = RelabelingProduct::create($request->all());

        // Log the action
        AuditLog::logAction(
            'create_relabeling_product',
            $product,
            Auth::user(),
            [],
            $product->toArray(),
            "Nuevo producto de reetiquetado creado"
        );

        return redirect()->route('relabeling.index')
            ->with('success', 'Producto de reetiquetado creado exitosamente.');
    }

    /**
     * Display the specified resource.
     */
    public function show(RelabelingProduct $relabeling)
    {
        return view('relabeling.show', compact('relabeling'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(RelabelingProduct $relabeling)
    {
        return view('relabeling.edit', compact('relabeling'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, RelabelingProduct $relabeling)
    {
        $request->validate([
            'proveedor_code' => 'required|string|max:50|unique:relabeling_products,proveedor_code,' . $relabeling->id,
            'codebar_gnu' => 'required|string|max:50',
            'description' => 'required|string',
            'price_cfa' => 'nullable|numeric',
        ]);

        $oldData = $relabeling->toArray();
        $relabeling->update($request->all());

        // Log the action
        AuditLog::logAction(
            'update_relabeling_product',
            $relabeling,
            Auth::user(),
            $oldData,
            $relabeling->toArray(),
            "Producto de reetiquetado actualizado"
        );

        return redirect()->route('relabeling.index')
            ->with('success', 'Producto de reetiquetado actualizado exitosamente.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(RelabelingProduct $relabeling)
    {
        $productData = $relabeling->toArray();
        $relabeling->delete();

        // Log the action
        AuditLog::logAction(
            'delete_relabeling_product',
            $relabeling,
            Auth::user(),
            $productData,
            [],
            "Producto de reetiquetado eliminado"
        );

        return redirect()->route('relabeling.index')
            ->with('success', 'Producto de reetiquetado eliminado exitosamente.');
    }

    /**
     * Search for relabeling product by provider code.
     */
    public function searchProduct(Request $request): JsonResponse
    {
        $request->validate([
            'proveedor_code' => 'required|string|max:50',
        ]);

        $product = RelabelingProduct::byProviderCode($request->proveedor_code)->first();

        if ($product) {
            return response()->json([
                'success' => true,
                'product' => $product,
                'message' => 'Producto encontrado.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Producto no encontrado.',
        ], 404);
    }

    /**
     * Save relabeling product.
     */
    public function saveProduct(Request $request): JsonResponse
    {
        $request->validate([
            'proveedor_code' => 'required|string|max:50',
            'codebar_gnu' => 'required|string|max:50',
            'description' => 'required|string',
            'price_cfa' => 'nullable|numeric',
        ]);

        try {
            $product = RelabelingProduct::updateOrCreate(
                ['proveedor_code' => $request->proveedor_code],
                [
                    'codebar_gnu' => $request->codebar_gnu,
                    'description' => $request->description,
                    'price_cfa' => $request->price_cfa,
                ]
            );

            // Log the action
            AuditLog::logAction(
                'save_relabeling_product',
                $product,
                Auth::user(),
                [],
                $product->toArray(),
                "Producto de reetiquetado guardado: {$request->proveedor_code}"
            );

            return response()->json([
                'success' => true,
                'product' => $product,
                'message' => 'Producto guardado con éxito.',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al guardar el producto: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generate ZPL content for label printing.
     */
    public function generateZpl(Request $request): JsonResponse
    {
        $request->validate([
            'barcode' => 'required|string|max:50',
            'my_code' => 'required|string|max:50',
            'description' => 'required|string',
            'price' => 'nullable|string',
        ]);

        $zpl = $this->createZplContent([
            'barcode' => $request->barcode,
            'my_code' => $request->my_code,
            'desc' => $request->description,
            'price' => $request->price ?? '0',
        ]);

        return response()->json([
            'success' => true,
            'zpl_content' => $zpl,
            'message' => 'Contenido ZPL generado.',
        ]);
    }

    /**
     * Create ZPL content for Zebra printer.
     */
    private function createZplContent(array $data): string
    {
        $zpl = '^XA';
        
        // Description
        $zpl .= '^FO20,20^ADN,36,20^FD' . $data['desc'] . '^FS';
        
        // Internal Code
        $zpl .= '^FO20,65^ADN,18,10^FDCÓDIGO INT: ' . $data['my_code'] . '^FS';
        
        // Price
        $zpl .= '^FO20,100^A0N,60,60^FDCFA ' . $data['price'] . '^FS';
        
        // Barcode
        $zpl .= '^FO20,170^BY3^BCN,100,Y,N,N^FD' . $data['barcode'] . '^FS';
        
        $zpl .= '^XZ';
        
        return $zpl;
    }

    /**
     * Get all relabeling products.
     */
    public function getAllProducts(): JsonResponse
    {
        $products = RelabelingProduct::orderBy('proveedor_code')->get();

        return response()->json([
            'success' => true,
            'products' => $products,
        ]);
    }

    /**
     * Delete relabeling product.
     */
    public function deleteProduct(Request $request): JsonResponse
    {
        $request->validate([
            'id' => 'required|exists:relabeling_products,id',
        ]);

        try {
            $product = RelabelingProduct::findOrFail($request->id);
            $productData = $product->toArray();
            
            $product->delete();

            // Log the action
            AuditLog::logAction(
                'delete_relabeling_product',
                $product,
                Auth::user(),
                $productData,
                [],
                "Producto de reetiquetado eliminado: {$productData['proveedor_code']}"
            );

            return response()->json([
                'success' => true,
                'message' => 'Producto eliminado correctamente.',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el producto: ' . $e->getMessage(),
            ], 500);
        }
    }
}