<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class QZTrayController extends Controller
{
    /**
     * Firmar datos para QZ Tray usando la llave privada
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function sign(Request $request)
    {
        try {
            $dataToSign = $request->input('data');
            
            if (empty($dataToSign)) {
                return response()->json([
                    'success' => false,
                    'error' => 'No se proporcionaron datos para firmar'
                ], 400);
            }

            // Ruta a la llave privada (debe estar en storage/app/private, no en public)
            $privateKeyPath = storage_path('app/private/private-key-pkcs8.pem');
            
            if (!file_exists($privateKeyPath)) {
                Log::error('QZ Tray: Llave privada no encontrada en: ' . $privateKeyPath);
                return response()->json([
                    'success' => false,
                    'error' => 'Llave privada no configurada. Contacte al administrador.'
                ], 500);
            }

            // Leer la llave privada
            $privateKey = file_get_contents($privateKeyPath);
            
            if ($privateKey === false) {
                Log::error('QZ Tray: No se pudo leer la llave privada');
                return response()->json([
                    'success' => false,
                    'error' => 'Error al leer la llave privada'
                ], 500);
            }

            // Convertir los datos a string si es necesario
            $dataString = is_array($dataToSign) ? json_encode($dataToSign) : (string)$dataToSign;

            // Firmar los datos
            $signature = '';
            $signed = openssl_sign($dataString, $signature, $privateKey, OPENSSL_ALGO_SHA512);
            
            if (!$signed) {
                Log::error('QZ Tray: Error al firmar datos. OpenSSL error: ' . openssl_error_string());
                return response()->json([
                    'success' => false,
                    'error' => 'Error al firmar los datos'
                ], 500);
            }

            // Convertir la firma a base64 para enviarla
            $signatureBase64 = base64_encode($signature);

            return response()->json([
                'success' => true,
                'signature' => $signatureBase64
            ]);

        } catch (\Exception $e) {
            Log::error('QZ Tray: Error en sign: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Error interno al firmar: ' . $e->getMessage()
            ], 500);
        }
    }
}

