<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Client;
use App\Models\Inventory;
use App\Models\AuditLog;
use App\Models\ContainerEntry;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderPreparationController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Display the order preparation form.
     */
    public function index()
    {
        $this->authorizeAction('view');
        
        // Obtener el contenedor asignado al usuario autenticado
        $assignedContainer = null;
        $operarioId = null;
        $user = Auth::user();

        if ($user) {
            // Usar la relación del modelo User para obtener el contenedor asignado
            $assignedContainer = $user->assignedContainer();

            // Si hay contenedor asignado, obtener el operario_id del usuario asignado
            if ($assignedContainer) {
                // Obtener el primer usuario asignado del contenedor
                $assignedUser = $assignedContainer->usuariosAsignados()->first();
                if (!$assignedUser && $assignedContainer->usuarioAsignado) {
                    $assignedUser = $assignedContainer->usuarioAsignado;
                }
                
                if ($assignedUser) {
                    $operarioId = $assignedUser->operario_id ?? $assignedUser->name;
                } else {
                    // Si no se encuentra usuario asignado, usar el operario_id del usuario actual
                    $operarioId = $user->operario_id ?? $user->name;
                }
            } else {
                // Si no hay contenedor, usar el operario_id del usuario actual
                $operarioId = $user->operario_id ?? $user->name;
            }
        }

        return view('order-preparation.index', [
            'assignedContainer' => $assignedContainer,
            'operarioId' => $operarioId
        ]);
    }

    /**
     * Get clients for Select2 dropdown.
     */
    public function getClients()
    {
        $this->authorizeAction('view');
        
        $clients = Client::select('id as client_id', 'client_name')
            ->where('is_active', true)
            ->orderBy('client_name')
            ->get();

        return response()->json($clients);
    }

    /**
     * Validate if a box number or MOCACO exists in inventory.
     */
    public function validateBoxOrMocaco(Request $request)
    {
        $this->authorizeAction('view');
        
        $input = trim($request->input('input'));
        
        if (empty($input)) {
            return response()->json([
                'exists' => false,
                'message' => 'El valor está vacío'
            ]);
        }

        // First, try to find by box number
        $inventoryCount = Inventory::byBox($input)
            ->where('status', 'disponible')
            ->count();
        
        // If no items found by box, try by MOCACO
        if ($inventoryCount === 0) {
            $inventoryCount = Inventory::where('mocaco', $input)
                ->where('status', 'disponible')
                ->count();
        }
        
        if ($inventoryCount > 0) {
            return response()->json([
                'exists' => true,
                'count' => $inventoryCount,
                'message' => "Encontrado: {$inventoryCount} item(s) disponible(s)"
            ]);
        } else {
            return response()->json([
                'exists' => false,
                'message' => 'No existe en inventario disponible'
            ]);
        }
    }

    /**
     * Save a draft order.
     */
    public function saveDraft(Request $request)
    {
        $this->authorizeAction('create');
        
        $request->validate([
            'client_id' => 'required|exists:clients,id',
            'operario_id' => 'required|string|max:50',
            'notes' => 'nullable|string',
            'boxes' => 'required|array|min:1',
            'boxes.*' => 'string|max:50',
        ]);

        DB::beginTransaction();

        try {
            // Create the order
            $order = Order::create([
                'client_id' => $request->client_id,
                'user_id' => Auth::id() ?? 1, // Default to user 1 if not authenticated
                'operario_id' => $request->operario_id,
                'notes' => $request->notes,
                'status' => 'draft',
                'order_date' => now(),
            ]);
            $boxesProcessed = 0;
            $itemsFound = 0;
            $itemsReserved = 0;
            
            foreach ($request->boxes as $rawInput) {
                $input = trim($rawInput);
                if ($input === '') { continue; }
                
                // Try to find by box number first
                $inventoryItems = Inventory::byBox($input)
                    ->where('status', 'disponible')
                    ->get();
                
                // If no items found by box, try by MOCACO
                if ($inventoryItems->isEmpty()) {
                    $inventoryItems = Inventory::where('mocaco', $input)
                        ->where('status', 'disponible')
                        ->get();
                }

                
                $boxesProcessed++;
                $itemsFound += $inventoryItems->count();
                
                foreach ($inventoryItems as $item) {
                    // Create OrderItem
                    OrderItem::create([
                        'order_id' => $order->id,
                        'inventory_id' => $item->id,
                        'full_barcode' => $item->full_barcode,
                        'mocaco' => $item->mocaco,
                        'n_carton' => $item->n_carton,
                    ]);
                    
                    // RESERVE the inventory item
                    $item->update([
                        'status' => 'reservado',
                        'order_id' => $order->id,
                    ]);
                    
                    $itemsReserved++;
                }
            }

            // // Add items from scanned boxes and RESERVE them
            // $boxesProcessed = 0;
            // $itemsFound = 0;
            // $itemsReserved = 0;

            // foreach ($request->boxes as $rawBoxNumber) {
            //     $boxNumber = trim($rawBoxNumber);
            //     if ($boxNumber === '') { continue; }

            //     $inventoryItems = Inventory::byBox($boxNumber)
            //         ->where('status', 'disponible')
            //         ->get();

            //     $boxesProcessed++;
            //     $itemsFound += $inventoryItems->count();

            //     foreach ($inventoryItems as $item) {
            //         // Create OrderItem
            //         OrderItem::create([
            //             'order_id' => $order->id,
            //             'inventory_id' => $item->id,
            //             'full_barcode' => $item->full_barcode,
            //             'mocaco' => $item->mocaco,
            //             'n_carton' => $item->n_carton,
            //         ]);

            //         // RESERVE the inventory item (set status to 'reservado' and link order_id)
            //         $updated = $item->update([
            //             'status' => 'reservado',
            //             'order_id' => $order->id,
            //         ]);
            //         if ($updated) { $itemsReserved++; }
            //     }
            // }

            // Log the action
            AuditLog::logAction(
                'create_draft_order',
                $order,
                Auth::user() ?? \App\Models\User::find(1),
                [],
                $order->toArray(),
                "Borrador de pedido creado con " . count($request->boxes) . " cajas"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'order_id' => $order->id,
                'message' => 'Borrador guardado con éxito.',
                'items_count' => $order->orderItems()->count(),
                'debug' => [
                    'boxes_processed' => $boxesProcessed,
                    'items_found' => $itemsFound,
                    'items_reserved' => $itemsReserved,
                    'boxes' => $request->boxes,
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al guardar el borrador: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get draft orders.
     */
    public function getDraftOrders()
    {
        $this->authorizeAction('view');
        
        $orders = Order::with(['client', 'orderItems'])
            ->where('status', 'draft')
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json($orders);
    }

    /**
     * Confirm and dispatch an order.
     */
    public function confirmDispatch(Request $request)
    {
        $this->authorizeAction('dispatch');
        
        $request->validate([
            'order_id' => 'required|exists:orders,id',
        ]);

        $order = Order::with('orderItems')->findOrFail($request->order_id);

        if ($order->status !== 'draft') {
            return response()->json([
                'success' => false,
                'message' => 'Solo se pueden despachar pedidos en estado borrador.'
            ], 400);
        }

        DB::beginTransaction();

        try {
            // Remove items from inventory
            $inventoryIds = $order->orderItems->pluck('inventory_id')->filter();
            Inventory::whereIn('id', $inventoryIds)->delete();

            // Update order status
            $order->dispatch();

            // Log the action
            AuditLog::logAction(
                'dispatch_order',
                $order,
                Auth::user() ?? \App\Models\User::find(1),
                $order->getOriginal(),
                $order->toArray(),
                "Pedido despachado con " . $order->orderItems()->count() . " artículos"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pedido confirmado y enviado con éxito.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al despachar el pedido: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get client order history.
     */
    public function getClientOrders($clientId)
    {
        $this->authorizeAction('view');
        
        $orders = Order::with('orderItems')
            ->where('client_id', $clientId)
            ->where('status', 'dispatched')
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json($orders);
    }

    /**
     * Get reserved orders (pedidos en estado 'reserved').
     */
    public function getReservedOrders()
    {
        $this->authorizeAction('view');
        
        $orders = Order::with(['client', 'orderItems'])
            ->where('status', 'reserved')
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json($orders);
    }

    /**
     * Load a reserved order with its boxes.
     */
    public function loadReservedOrder($orderId)
    {
        $this->authorizeAction('view');
        
        $order = Order::with(['client', 'orderItems.inventory'])
            ->where('status', 'reserved')
            ->findOrFail($orderId);

        // Get unique box numbers from order items
        $boxes = $order->orderItems()
            ->select('n_carton')
            ->distinct()
            ->pluck('n_carton')
            ->filter()
            ->values();

        return response()->json([
            'success' => true,
            'order' => $order,
            'boxes' => $boxes,
            'total_boxes' => $boxes->count(),
            'total_items' => $order->orderItems()->count(),
        ]);
    }

    /**
     * Confirm a reserved order by physically scanning boxes.
     */
    public function confirmReservedOrder(Request $request)
    {
        $this->authorizeAction('dispatch');
        
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'scanned_boxes' => 'required|array|min:1',
            'scanned_boxes.*' => 'string|max:50',
            'extra_boxes' => 'nullable|array',
            'extra_boxes.*' => 'string|max:50',
        ]);

        $order = Order::with('orderItems')->findOrFail($request->order_id);

        if ($order->status !== 'reserved') {
            return response()->json([
                'success' => false,
                'message' => 'Solo se pueden confirmar pedidos en estado reservado.'
            ], 400);
        }

        DB::beginTransaction();

        try {
            // Get reserved boxes from order
            $reservedBoxes = $order->orderItems()
                ->select('n_carton')
                ->distinct()
                ->pluck('n_carton')
                ->filter()
                ->map(fn($box) => trim($box))
                ->toArray();

            $scannedBoxes = array_map('trim', $request->scanned_boxes);
            $extraBoxes = $request->filled('extra_boxes') ? array_map('trim', $request->extra_boxes) : [];

            // Validate that all reserved boxes are scanned
            $missingBoxes = array_diff($reservedBoxes, $scannedBoxes);
            if (!empty($missingBoxes)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Faltan cajas por escanear: ' . implode(', ', $missingBoxes),
                    'missing_boxes' => array_values($missingBoxes),
                ], 400);
            }

            // Validate scanned boxes are in the reserved order
            $invalidBoxes = array_diff($scannedBoxes, $reservedBoxes);
            if (!empty($invalidBoxes)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Algunas cajas escaneadas no están en el pedido reservado: ' . implode(', ', $invalidBoxes),
                    'invalid_boxes' => array_values($invalidBoxes),
                ], 400);
            }

            // Process extra boxes if any
            $extraItemsReserved = 0;
            if (!empty($extraBoxes)) {
                foreach ($extraBoxes as $boxNumber) {
                    if (empty($boxNumber)) continue;
                    
                    // Find available inventory items for extra box
                    $inventoryItems = Inventory::byBox($boxNumber)
                        ->where('status', 'disponible')
                        ->get();
                    
                    foreach ($inventoryItems as $item) {
                        // Create OrderItem for extra box
                        OrderItem::create([
                            'order_id' => $order->id,
                            'inventory_id' => $item->id,
                            'full_barcode' => $item->full_barcode,
                            'mocaco' => $item->mocaco,
                            'n_carton' => $item->n_carton,
                        ]);
                        
                        // Reserve the inventory item
                        $item->update([
                            'status' => 'reservado',
                            'order_id' => $order->id,
                        ]);
                        
                        $extraItemsReserved++;
                    }
                }
            }

            // Update all inventory items from reserved boxes to 'dispatched'
            $inventoryIds = $order->orderItems->pluck('inventory_id')->filter();
            Inventory::whereIn('id', $inventoryIds)
                ->update([
                    'status' => 'dispatched',
                ]);

            // Update order status to dispatched
            $order->update(['status' => 'dispatched']);

            // Log the action
            AuditLog::logAction(
                'confirm_reserved_order',
                $order,
                Auth::user() ?? \App\Models\User::find(1),
                ['status' => 'reserved'],
                ['status' => 'dispatched'],
                "Pedido reservado confirmado físicamente. Cajas escaneadas: " . count($scannedBoxes) . 
                ($extraItemsReserved > 0 ? " | Cajas extra añadidas: {$extraItemsReserved} unidades" : "")
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pedido reservado confirmado y despachado con éxito.',
                'scanned_boxes' => count($scannedBoxes),
                'extra_items' => $extraItemsReserved,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al confirmar el pedido: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Validate if a scanned box belongs to a reserved order.
     */
    public function validateReservedBox(Request $request)
    {
        $this->authorizeAction('view');
        
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'box_number' => 'required|string|max:50',
        ]);

        $order = Order::findOrFail($request->order_id);
        $boxNumber = trim($request->box_number);

        // Check if box is in the reserved order
        $isInOrder = $order->orderItems()
            ->where('n_carton', $boxNumber)
            ->exists();

        // Check if box is available (for extra boxes)
        $isAvailable = Inventory::byBox($boxNumber)
            ->where('status', 'disponible')
            ->exists();

        return response()->json([
            'is_in_order' => $isInOrder,
            'is_available' => $isAvailable,
            'message' => $isInOrder 
                ? 'Caja confirmada correctamente' 
                : ($isAvailable ? 'Caja disponible para añadir como extra' : 'Caja no encontrada o no disponible'),
        ]);
    }
}