<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Client;
use App\Models\Inventory;
use App\Models\AuditLog;
use App\Models\ContainerEntry;
use App\Models\BoxReservation;
use App\Models\BoxReservationItem;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class OrderPreparationController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Display the order preparation form.
     */
    public function index()
    {
        $this->authorizeAction('view');

        // Obtener el contenedor asignado al usuario autenticado
        $assignedContainer = null;
        $operarioId = null;
        $user = Auth::user();

        if ($user) {
            // Usar la relación del modelo User para obtener el contenedor asignado
            $assignedContainer = $user->assignedContainer();

            // Si hay contenedor asignado, obtener el operario_id del usuario asignado
            if ($assignedContainer) {
                // Obtener el primer usuario asignado del contenedor
                $assignedUser = $assignedContainer->usuariosAsignados()->first();
                if (!$assignedUser && $assignedContainer->usuarioAsignado) {
                    $assignedUser = $assignedContainer->usuarioAsignado;
                }

                if ($assignedUser) {
                    $operarioId = $assignedUser->operario_id ?? $assignedUser->name;
                } else {
                    // Si no se encuentra usuario asignado, usar el operario_id del usuario actual
                    $operarioId = $user->operario_id ?? $user->name;
                }
            } else {
                // Si no hay contenedor, usar el operario_id del usuario actual
                $operarioId = $user->operario_id ?? $user->name;
            }
        }

        return view('order-preparation.index', [
            'assignedContainer' => $assignedContainer,
            'operarioId' => $operarioId
        ]);
    }

    /**
     * Get clients for Select2 dropdown.
     */
    public function getClients()
    {
        $this->authorizeAction('view');

        $clients = Client::select('id as client_id', 'client_name')
            ->where('is_active', true)
            ->orderBy('client_name')
            ->get();

        return response()->json($clients);
    }

    /**
     * Validate if a box number or MOCACO exists in inventory.
     */
    public function validateBoxOrMocaco(Request $request)
    {
        $this->authorizeAction('view');

        $input = trim($request->input('input'));

        if (empty($input)) {
            return response()->json([
                'exists' => false,
                'message' => 'El valor está vacío'
            ]);
        }

        // First, try to find by box number
        $inventoryCount = Inventory::byBox($input)
            ->where('status', 'disponible')
            ->count();

        // If no items found by box, try by MOCACO
        if ($inventoryCount === 0) {
            $inventoryCount = Inventory::where('mocaco', $input)
                ->where('status', 'disponible')
                ->count();
        }

        if ($inventoryCount > 0) {
            return response()->json([
                'exists' => true,
                'count' => $inventoryCount,
                'message' => "Encontrado: {$inventoryCount} item(s) disponible(s)"
            ]);
        } else {
            return response()->json([
                'exists' => false,
                'message' => 'No existe en inventario disponible'
            ]);
        }
    }

    /**
     * Save a draft order.
     */
    public function saveDraft(Request $request)
    {
        $this->authorizeAction('create');

        $request->validate([
            'client_id' => 'required|exists:clients,id',
            'operario_id' => 'required|string|max:50',
            'notes' => 'nullable|string',
            'boxes' => 'required|array|min:1',
            'boxes.*' => 'string|max:50',
        ]);

        DB::beginTransaction();

        try {
            // Create the order
            $order = Order::create([
                'client_id' => $request->client_id,
                'user_id' => Auth::id() ?? 1, // Default to user 1 if not authenticated
                'operario_id' => $request->operario_id,
                'notes' => $request->notes,
                'status' => 'draft',
                'order_date' => now(),
            ]);
            $boxesProcessed = 0;
            $itemsFound = 0;
            $itemsReserved = 0;

            foreach ($request->boxes as $rawInput) {
                $input = trim($rawInput);
                if ($input === '') {
                    continue;
                }

                // Try to find by box number first
                $inventoryItems = Inventory::byBox($input)
                    ->where('status', 'disponible')
                    ->get();

                // If no items found by box, try by MOCACO
                if ($inventoryItems->isEmpty()) {
                    $inventoryItems = Inventory::where('mocaco', $input)
                        ->where('status', 'disponible')
                        ->get();
                }


                $boxesProcessed++;
                $itemsFound += $inventoryItems->count();

                foreach ($inventoryItems as $item) {
                    // Create OrderItem
                    OrderItem::create([
                        'order_id' => $order->id,
                        'inventory_id' => $item->id,
                        'full_barcode' => $item->full_barcode,
                        'mocaco' => $item->mocaco,
                        'n_carton' => $item->n_carton,
                    ]);

                    // RESERVE the inventory item
                    $item->update([
                        'status' => 'reservado',
                        'order_id' => $order->id,
                    ]);

                    $itemsReserved++;
                }
            }

            // // Add items from scanned boxes and RESERVE them
            // $boxesProcessed = 0;
            // $itemsFound = 0;
            // $itemsReserved = 0;

            // foreach ($request->boxes as $rawBoxNumber) {
            //     $boxNumber = trim($rawBoxNumber);
            //     if ($boxNumber === '') { continue; }

            //     $inventoryItems = Inventory::byBox($boxNumber)
            //         ->where('status', 'disponible')
            //         ->get();

            //     $boxesProcessed++;
            //     $itemsFound += $inventoryItems->count();

            //     foreach ($inventoryItems as $item) {
            //         // Create OrderItem
            //         OrderItem::create([
            //             'order_id' => $order->id,
            //             'inventory_id' => $item->id,
            //             'full_barcode' => $item->full_barcode,
            //             'mocaco' => $item->mocaco,
            //             'n_carton' => $item->n_carton,
            //         ]);

            //         // RESERVE the inventory item (set status to 'reservado' and link order_id)
            //         $updated = $item->update([
            //             'status' => 'reservado',
            //             'order_id' => $order->id,
            //         ]);
            //         if ($updated) { $itemsReserved++; }
            //     }
            // }

            // Log the action
            AuditLog::logAction(
                'create_draft_order',
                $order,
                Auth::user() ?? \App\Models\User::find(1),
                [],
                $order->toArray(),
                "Borrador de pedido creado con " . count($request->boxes) . " cajas"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'order_id' => $order->id,
                'message' => 'Borrador guardado con éxito.',
                'items_count' => $order->orderItems()->count(),
                'debug' => [
                    'boxes_processed' => $boxesProcessed,
                    'items_found' => $itemsFound,
                    'items_reserved' => $itemsReserved,
                    'boxes' => $request->boxes,
                ]
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al guardar el borrador: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get draft orders.
     */
    public function getDraftOrders()
    {
        $this->authorizeAction('view');

        $orders = Order::with(['client', 'orderItems'])
            ->where('status', 'draft')
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json($orders);
    }

    /**
     * Confirm and dispatch an order.
     */
    public function confirmDispatch(Request $request)
    {
        $this->authorizeAction('dispatch');

        $request->validate([
            'order_id' => 'required|exists:orders,id',
        ]);

        $order = Order::with('orderItems')->findOrFail($request->order_id);

        if ($order->status !== 'draft') {
            return response()->json([
                'success' => false,
                'message' => 'Solo se pueden despachar pedidos en estado borrador.'
            ], 400);
        }

        DB::beginTransaction();

        try {
            // Remove items from inventory
            $inventoryIds = $order->orderItems->pluck('inventory_id')->filter();
            Inventory::whereIn('id', $inventoryIds)->delete();

            // Update order status
            $order->dispatch();

            // Log the action
            AuditLog::logAction(
                'dispatch_order',
                $order,
                Auth::user() ?? \App\Models\User::find(1),
                $order->getOriginal(),
                $order->toArray(),
                "Pedido despachado con " . $order->orderItems()->count() . " artículos"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pedido confirmado y enviado con éxito.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al despachar el pedido: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get client order history.
     */
    public function getClientOrders($clientId)
    {
        $this->authorizeAction('view');

        $orders = Order::with('orderItems')
            ->where('client_id', $clientId)
            ->where('status', 'dispatched')
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json($orders);
    }

    /**
     * Get reserved orders (pedidos en estado 'reserved').
     */
    public function getReservedOrders()
    {
        $this->authorizeAction('view');

        $orders = Order::with(['client', 'orderItems'])
            ->where('status', 'reserved')
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json($orders);
    }

    /**
     * Load a reserved order with its boxes.
     */
    public function loadReservedOrder($orderId)
    {
        $this->authorizeAction('view');

        $order = Order::with(['client', 'orderItems.inventory'])
            ->where('status', 'reserved')
            ->findOrFail($orderId);

        // Get unique box numbers from order items
        $boxes = $order->orderItems()
            ->select('n_carton')
            ->distinct()
            ->pluck('n_carton')
            ->filter()
            ->values();

        return response()->json([
            'success' => true,
            'order' => $order,
            'boxes' => $boxes,
            'total_boxes' => $boxes->count(),
            'total_items' => $order->orderItems()->count(),
        ]);
    }

    /**
     * Confirm a reserved order by physically scanning boxes.
     */
    public function confirmReservedOrder(Request $request)
    {
        $this->authorizeAction('dispatch');

        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'scanned_boxes' => 'required|array|min:1',
            'scanned_boxes.*' => 'string|max:50',
            'extra_boxes' => 'nullable|array',
            'extra_boxes.*' => 'string|max:50',
        ]);

        $order = Order::with('orderItems')->findOrFail($request->order_id);

        if ($order->status !== 'reserved') {
            return response()->json([
                'success' => false,
                'message' => 'Solo se pueden confirmar pedidos en estado reservado.'
            ], 400);
        }

        DB::beginTransaction();

        try {
            // Get reserved boxes from order
            $reservedBoxes = $order->orderItems()
                ->select('n_carton')
                ->distinct()
                ->pluck('n_carton')
                ->filter()
                ->map(fn($box) => trim($box))
                ->toArray();

            $scannedBoxes = array_map('trim', $request->scanned_boxes);
            $extraBoxes = $request->filled('extra_boxes') ? array_map('trim', $request->extra_boxes) : [];

            // Validate that all reserved boxes are scanned
            $missingBoxes = array_diff($reservedBoxes, $scannedBoxes);
            if (!empty($missingBoxes)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Faltan cajas por escanear: ' . implode(', ', $missingBoxes),
                    'missing_boxes' => array_values($missingBoxes),
                ], 400);
            }

            // Validate scanned boxes are in the reserved order
            $invalidBoxes = array_diff($scannedBoxes, $reservedBoxes);
            if (!empty($invalidBoxes)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Algunas cajas escaneadas no están en el pedido reservado: ' . implode(', ', $invalidBoxes),
                    'invalid_boxes' => array_values($invalidBoxes),
                ], 400);
            }

            // Process extra boxes if any
            $extraItemsReserved = 0;
            if (!empty($extraBoxes)) {
                foreach ($extraBoxes as $boxNumber) {
                    if (empty($boxNumber)) continue;

                    // Find available inventory items for extra box
                    $inventoryItems = Inventory::byBox($boxNumber)
                        ->where('status', 'disponible')
                        ->get();

                    foreach ($inventoryItems as $item) {
                        // Create OrderItem for extra box
                        OrderItem::create([
                            'order_id' => $order->id,
                            'inventory_id' => $item->id,
                            'full_barcode' => $item->full_barcode,
                            'mocaco' => $item->mocaco,
                            'n_carton' => $item->n_carton,
                        ]);

                        // Reserve the inventory item
                        $item->update([
                            'status' => 'reservado',
                            'order_id' => $order->id,
                        ]);

                        $extraItemsReserved++;
                    }
                }
            }

            // Update all inventory items from reserved boxes to 'dispatched'
            $inventoryIds = $order->orderItems->pluck('inventory_id')->filter();
            Inventory::whereIn('id', $inventoryIds)
                ->update([
                    'status' => 'dispatched',
                ]);

            // Update order status to dispatched
            $order->update(['status' => 'dispatched']);

            // Log the action
            AuditLog::logAction(
                'confirm_reserved_order',
                $order,
                Auth::user() ?? \App\Models\User::find(1),
                ['status' => 'reserved'],
                ['status' => 'dispatched'],
                "Pedido reservado confirmado físicamente. Cajas escaneadas: " . count($scannedBoxes) .
                    ($extraItemsReserved > 0 ? " | Cajas extra añadidas: {$extraItemsReserved} unidades" : "")
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pedido reservado confirmado y despachado con éxito.',
                'scanned_boxes' => count($scannedBoxes),
                'extra_items' => $extraItemsReserved,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al confirmar el pedido: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Validate if a scanned box belongs to a reserved order.
     */
    public function validateReservedBox(Request $request)
    {
        $this->authorizeAction('view');

        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'box_number' => 'required|string|max:50',
        ]);

        $order = Order::findOrFail($request->order_id);
        $boxNumber = trim($request->box_number);

        // Check if box is in the reserved order
        $isInOrder = $order->orderItems()
            ->where('n_carton', $boxNumber)
            ->exists();

        // Check if box is available (for extra boxes)
        $isAvailable = Inventory::byBox($boxNumber)
            ->where('status', 'disponible')
            ->exists();

        return response()->json([
            'is_in_order' => $isInOrder,
            'is_available' => $isAvailable,
            'message' => $isInOrder
                ? 'Caja confirmada correctamente'
                : ($isAvailable ? 'Caja disponible para añadir como extra' : 'Caja no encontrada o no disponible'),
        ]);
    }

    /**
     * Get active box reservations for order preparation
     */
    public function getBoxReservations()
    {
        $reservations = BoxReservation::with(['client', 'items'])
            ->whereIn('status', ['pending', 'in_progress'])
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($reservation) {
                return [
                    'id' => $reservation->id,
                    'reservation_code' => $reservation->reservation_code,
                    'client_name' => $reservation->client->client_name ?? 'Cliente no asignado',
                    'client_id' => $reservation->client_id,
                    'status' => $reservation->status,
                    'total_boxes' => $reservation->total_boxes,
                    'confirmed_boxes' => $reservation->confirmed_boxes,
                    'pending_boxes' => $reservation->pending_boxes,
                    'completion_percentage' => $reservation->completion_percentage,
                    'created_at' => $reservation->created_at->format('Y-m-d H:i'),
                ];
            });

        return response()->json($reservations);
    }

    /**
     * Start processing a box reservation
     */
    public function startBoxReservation(Request $request)
    {
        $request->validate([
            'reservation_id' => 'required|exists:box_reservations,id',
        ]);

        $reservation = BoxReservation::with(['client', 'items.inventory'])->findOrFail($request->reservation_id);

        // Update status to in_progress if pending
        if ($reservation->isPending()) {
            $reservation->update(['status' => 'in_progress']);
        }

        // Get all reservation items with inventory details
        $boxes = $reservation->items->map(function ($item) {
            return [
                'id' => $item->id,
                'box_number' => $item->box_number,
                'quantity' => $item->quantity,
                'status' => $item->status,
                'scanned_at' => $item->scanned_at?->format('Y-m-d H:i:s'),
                'inventory' => $item->inventory ? [
                    'id' => $item->inventory->id,
                    'mocaco' => $item->inventory->mocaco,
                    'marca' => $item->inventory->cadena,
                    'seccion' => $item->inventory->seccion,
                    'familia' => $item->inventory->familia_articulo_description,
                    'ubicacion' => $item->inventory->ubicacion,
                ] : null,
            ];
        });

        return response()->json([
            'success' => true,
            'message' => 'Reserva cargada correctamente',
            'reservation' => [
                'id' => $reservation->id,
                'reservation_code' => $reservation->reservation_code,
                'client_name' => $reservation->client?->client_name ?? 'Cliente no encontrado',
                'client_id' => $reservation->client_id,
                'status' => $reservation->status,
                'notes' => $reservation->notes,
            ],
            'boxes' => $boxes,
            'stats' => [
                'total' => $reservation->total_boxes,
                'confirmed' => $reservation->confirmed_boxes,
                'pending' => $reservation->pending_boxes,
            ],
        ]);
    }

    /**
     * Confirm a scanned box from reservation
     */
    public function confirmBoxReservationScan(Request $request)
    {
        $request->validate([
            'reservation_id' => 'required|exists:box_reservations,id',
            'box_number' => 'required|string',
        ]);

        $reservation = BoxReservation::findOrFail($request->reservation_id);
        $boxNumber = trim($request->box_number);

        // Check if box is in this reservation
        $reservationItem = BoxReservationItem::where('reservation_id', $reservation->id)
            ->where('box_number', $boxNumber)
            ->where('status', 'reserved')
            ->first();

        if ($reservationItem) {
            // Box is in reservation - confirm it
            $reservationItem->confirm();

            // Update all related inventory items
            Inventory::where('n_carton', $boxNumber)
                ->where('reservation_id', $reservation->id) // Security check to only update items in this reservation if already linked
                ->update([
                    'reservation_status' => 'confirmed',
                    // 'status' is likely already 'order_proposal' from import, but ensuring consistency
                    'status' => 'order_proposal',
                ]);

            return response()->json([
                'success' => true,
                'status' => 'confirmed',
                'message' => 'Caja confirmada correctamente',
                'is_from_reservation' => true,
                'item' => $reservationItem,
                'stats' => [
                    'total' => $reservation->fresh()->total_boxes,
                    'confirmed' => $reservation->fresh()->confirmed_boxes,
                    'pending' => $reservation->fresh()->pending_boxes,
                ],
            ]);
        }

        // Box not in reservation - check if it's available
        $inventory = Inventory::where('n_carton', $boxNumber)
            ->where('reservation_status', 'available')
            ->first();

        if ($inventory) {
            return response()->json([
                'success' => true,
                'status' => 'not_in_reservation',
                'message' => '¿Desea agregar esta caja a la reserva?',
                'box' => [
                    'n_carton' => $inventory->n_carton,
                    'marca' => $inventory->cadena,
                    'seccion' => $inventory->seccion,
                    'ubicacion' => $inventory->ubicacion,
                ],
                'is_from_reservation' => false,
            ]);
        }

        return response()->json([
            'success' => false,
            'status' => 'error',
            'message' => 'Caja no encontrada o no disponible',
        ], 404);
    }

    /**
     * Add extra box to reservation (not originally in the list)
     */
    public function addExtraBoxToReservation(Request $request)
    {
        $request->validate([
            'reservation_id' => 'required|exists:box_reservations,id',
            'box_number' => 'required|string',
        ]);

        DB::beginTransaction();

        try {
            $reservation = BoxReservation::findOrFail($request->reservation_id);
            $boxNumber = trim($request->box_number);

            // First check if box exists at all
            $inventoryExists = Inventory::where('n_carton', $boxNumber)->exists();

            if (!$inventoryExists) {
                return response()->json([
                    'success' => false,
                    'status' => 'error',
                    'message' => "La caja {$boxNumber} no existe en el sistema",
                ], 404);
            }

            // Then check if it is available
            $inventory = Inventory::where('n_carton', $boxNumber)
                ->where('reservation_status', 'available')
                ->first();

            if (!$inventory) {
                return response()->json([
                    'success' => false,
                    'status' => 'error',
                    'message' => "La caja {$boxNumber} existe pero no está disponible (ya reservada o despachada)",
                ], 400);
            }

            // Create reservation item
            $item = BoxReservationItem::create([
                'reservation_id' => $reservation->id,
                'inventory_id' => $inventory->id,
                'box_number' => $boxNumber,
                'quantity' => 1,
                'status' => 'confirmed',
                'scanned_at' => now(),
                'notes' => 'Agregada por operario durante confirmación',
            ]);

            // Update all inventory items with this box number
            Inventory::where('n_carton', $boxNumber)
                ->where('reservation_status', 'available')
                ->update([
                    'reservation_status' => 'confirmed', // Assuming extra box added directly is confirmed? Or should it follow flow?
                    // Actually, if we are adding it as extra during scan, it IS confirmed.
                    'status' => 'order_proposal',
                    'reserved_for_client_id' => $reservation->client_id,
                    'reservation_id' => $reservation->id,
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'status' => 'added',
                'message' => 'Caja agregada a la reserva',
                'item' => $item,
                'stats' => [
                    'total' => $reservation->fresh()->total_boxes,
                    'confirmed' => $reservation->fresh()->confirmed_boxes,
                    'pending' => $reservation->fresh()->pending_boxes,
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error adding extra box to reservation', [
                'reservation_id' => $request->reservation_id,
                'box_number' => $request->box_number,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'status' => 'error',
                'message' => 'Error al agregar caja: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Complete box reservation and create order
     */
    public function completeBoxReservation(Request $request)
    {
        $request->validate([
            'reservation_id' => 'required|exists:box_reservations,id',
        ]);

        DB::beginTransaction();

        try {
            $reservation = BoxReservation::with(['items.inventory', 'client'])->findOrFail($request->reservation_id);

            // Validate that at least one box is confirmed
            if ($reservation->confirmed_boxes == 0) {
                return response()->json([
                    'success' => false,
                    'status' => 'error',
                    'message' => 'Debe confirmar al menos una caja antes de completar la reserva',
                ], 400);
            }

            // Generate order number
            $lastOrder = Order::orderBy('id', 'desc')->first();
            $orderNumber = 'ORD-' . date('Ymd') . '-' . str_pad(($lastOrder?->id ?? 0) + 1, 4, '0', STR_PAD_LEFT);

            // Create order
            $order = Order::create([
                'client_id' => $reservation->client_id,
                'user_id' => Auth::id(),
                'operario_id' => Auth::user()->operario_id ?? Auth::user()->name,
                'order_number' => $orderNumber,
                'status' => 'pending',
                'order_date' => now(),
                'dispatched_at' => null,
                'notes' => "Creado desde reserva {$reservation->reservation_code}",
            ]);

            // Get all confirmed items
            $confirmedItems = $reservation->items()->where('status', 'confirmed')->get();

            foreach ($confirmedItems as $item) {
                // Create order item
                OrderItem::create([
                    'order_id' => $order->id,
                    'inventory_id' => $item->inventory_id,
                    'full_barcode' => $item->inventory->full_barcode,
                    'mocaco' => $item->inventory->mocaco,
                    'n_carton' => $item->box_number,
                ]);

                // Update inventory status to reserved (since order is draft)
                $item->inventory->update([
                    'status' => 'reservado',
                    'client_id' => $reservation->client_id,
                    'order_id' => $order->id,
                ]);

                // Log audit
                AuditLog::logAction(
                    'dispatch_inventory',
                    $item->inventory,
                    Auth::user(),
                    ['status' => 'confirmed'],
                    ['status' => 'Despachado'],
                    "Despachado en orden {$orderNumber} desde reserva {$reservation->reservation_code}"
                );
            }

            // Update reservation status
            $reservation->update([
                'status' => 'completed',
                'confirmed_by_user_id' => Auth::id(),
                'confirmed_at' => now(),
            ]);

            // Log reservation completion
            AuditLog::logAction(
                'complete_reservation',
                $reservation,
                Auth::user(),
                ['status' => $reservation->getOriginal('status')],
                ['status' => 'completed'],
                "Reserva completada. Orden {$orderNumber} creada con {$confirmedItems->count()} cajas"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'status' => 'completed',
                'message' => "Reserva completada. Orden {$orderNumber} creada con {$confirmedItems->count()} cajas",
                'order' => [
                    'id' => $order->id,
                    'order_number' => $orderNumber,
                    'client_name' => $reservation->client->client_name,
                    'total_boxes' => $confirmedItems->count(),
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error completing box reservation', [
                'reservation_id' => $request->reservation_id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'status' => 'error',
                'message' => 'Error al completar reserva: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Cancel a box reservation
     */
    public function cancelBoxReservation(Request $request)
    {
        $request->validate([
            'reservation_id' => 'required|exists:box_reservations,id',
        ]);

        $reservation = BoxReservation::findOrFail($request->reservation_id);

        // Only allow cancellation for pending or in_progress reservations
        if ($reservation->isCompleted() || $reservation->isCancelled()) {
            return response()->json([
                'success' => false,
                'status' => 'error',
                'message' => 'No se puede cancelar una reserva completada o ya cancelada.',
            ], 400);
        }

        try {
            DB::beginTransaction();

            // Update reservation status
            $reservation->update(['status' => 'cancelled']);

            // Get inventory IDs from items
            $inventoryIds = BoxReservationItem::where('reservation_id', $reservation->id)
                ->pluck('inventory_id')
                ->toArray();

            // Cancel all items
            BoxReservationItem::where('reservation_id', $reservation->id)
                ->update(['status' => 'cancelled']);

            // Release inventory (by Item link OR by Reservation ID link)
            Inventory::where(function ($q) use ($inventoryIds, $reservation) {
                if (!empty($inventoryIds)) {
                    $q->whereIn('id', $inventoryIds);
                }
                $q->orWhere('reservation_id', $reservation->id);
            })->update([
                'reservation_status' => 'available',
                'status' => 'disponible',
                'reserved_for_client_id' => null,
                'reservation_id' => null,
            ]);

            // Also check for items that might be confirmed but not yet in an order (in case of partial flow)
            // The above query handles items linked by reservation_id.
            // If there were extra boxes added directly, they should have reservation_id set too (as per addExtraBox logic)

            DB::commit();

            return response()->json([
                'success' => true,
                'status' => 'cancelled',
                'message' => "Reserva cancelada correctamente. Las cajas han sido liberadas.",
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error cancelling reservation: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'status' => 'error',
                'message' => 'Error al cancelar la reserva: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove a box from reservation
     */
    public function removeBoxFromReservation(Request $request)
    {
        $request->validate([
            'reservation_id' => 'required|exists:box_reservations,id',
            'box_number' => 'required|string',
        ]);

        $reservation = BoxReservation::findOrFail($request->reservation_id);
        $boxNumber = trim($request->box_number);

        // Allow removing from pending or in_progress. Not completed/cancelled.
        if ($reservation->isCompleted() || $reservation->isCancelled()) {
            return response()->json([
                'success' => false,
                'status' => 'error',
                'message' => 'No se puede modificar una reserva completada o cancelada.',
            ], 400);
        }

        try {
            DB::beginTransaction();

            $item = BoxReservationItem::where('reservation_id', $reservation->id)
                ->where('box_number', $boxNumber)
                ->first();

            if (!$item) {
                return response()->json(['success' => false, 'message' => 'Caja no encontrada en la reserva'], 404);
            }

            // Release ALL inventory items with this box number that belong to this reservation
            $releasedCount = Inventory::where('n_carton', $boxNumber)
                ->where('reservation_id', $reservation->id)
                ->update([
                    'reservation_status' => 'available',
                    'status' => 'disponible',
                    'reserved_for_client_id' => null,
                    'reservation_id' => null,
                ]);

            // Delete item
            $item->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'status' => 'removed',
                'message' => "Caja {$boxNumber} eliminada de la reserva ({$releasedCount} unidades liberadas)",
                'stats' => [
                    'total' => $reservation->fresh()->total_boxes,
                    'confirmed' => $reservation->fresh()->confirmed_boxes,
                    'pending' => $reservation->fresh()->pending_boxes,
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Update reservation details (Client, Notes)
     */
    public function updateReservationDetails(Request $request)
    {
        $request->validate([
            'reservation_id' => 'required|exists:box_reservations,id',
            'client_id' => 'required|exists:clients,id',
            'notes' => 'nullable|string',
        ]);

        $reservation = BoxReservation::findOrFail($request->reservation_id);

        if ($reservation->isCompleted() || $reservation->isCancelled()) {
            return response()->json(['success' => false, 'message' => 'Reserva cerrada'], 400);
        }

        $reservation->update([
            'client_id' => $request->client_id,
            'notes' => $request->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Detalles actualizados correctamente',
            'client_name' => $reservation->client->client_name,
            'notes' => $reservation->notes,
        ]);
    }
}
