<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Inventory;
use App\Models\Client;
use App\Models\AuditLog;
use App\Exports\OrderDetailExport;
use App\Exports\OrdersListExport;
use App\Exports\OrderReservationTemplateExport;
use App\Imports\OrderReservationsImport;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Log;

class OrderController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAction('view');

        $query = Order::with(['client', 'user']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('id', 'like', "%{$search}%")
                    ->orWhereHas('client', function ($clientQuery) use ($search) {
                        $clientQuery->where('client_name', 'like', "%{$search}%");
                    });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('client_id')) {
            $query->where('client_id', $request->client_id);
        }

        $orders = $query->orderBy('created_at', 'desc')->paginate(20);
        $clients = Client::orderBy('client_name')->get();

        return view('orders.index', compact('orders', 'clients'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAction('create');

        $clients = Client::orderBy('client_name')->get();
        return view('orders.create', compact('clients'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAction('create');

        $request->validate([
            'client_id' => 'required|exists:clients,id',
            'operario_id' => 'nullable|string|max:20',
            'notes' => 'nullable|string',
            'status' => 'required|in:draft,reserved,pending,dispatched,cancelled',
        ]);

        $order = Order::create([
            'client_id' => $request->client_id,
            'user_id' => Auth::id(),
            'operario_id' => $request->operario_id,
            'notes' => $request->notes,
            'status' => $request->status,
            'order_date' => now(),
        ]);

        // Log the action
        AuditLog::logAction(
            'create_order',
            $order,
            Auth::user(),
            [],
            $order->toArray(),
            "Nuevo pedido creado"
        );

        return redirect()->route('orders.index')
            ->with('success', 'Pedido creado exitosamente.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        $this->authorizeAction('view');

        $order->load(['client', 'user', 'orderItems.inventory']);
        return view('orders.show', compact('order'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Order $order)
    {
        $this->authorizeAction('update');

        $clients = Client::orderBy('client_name')->get();
        return view('orders.edit', compact('order', 'clients'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Order $order)
    {
        $this->authorizeAction('update');

        $request->validate([
            'client_id' => 'required|exists:clients,id',
            'operario_id' => 'nullable|string|max:20',
            'notes' => 'nullable|string',
            'status' => 'required|in:draft,reserved,pending,dispatched,cancelled',
            'url_pdf_factura' => 'nullable|string|max:500',
            'factura_file' => 'nullable|file|mimes:pdf|max:10240', // 10MB máximo
        ]);

        // Verificar permisos para gestionar facturas
        $canManageInvoices = Auth::user()->hasRole('admin')
            || Auth::user()->hasRole('facturacion')
            || Auth::user()->can('orders.manage_invoices');

        $oldData = $order->toArray();
        $oldStatus = $order->status;

        $updateData = $request->except(['factura_file']);

        // Si se intenta subir o modificar factura, verificar permisos
        if (($request->hasFile('factura_file') || $request->filled('url_pdf_factura')) && !$canManageInvoices) {
            return back()->with('error', 'No tienes permisos para gestionar facturas.');
        }

        // Si se subió un archivo de factura, guardarlo
        if ($request->hasFile('factura_file') && $canManageInvoices) {
            $file = $request->file('factura_file');
            $name = 'pedido_' . $order->id . '_' . time() . '.' . $file->getClientOriginalExtension();
            // Guardar en el disco 'public' dentro de la carpeta 'facturas'
            $file->storeAs('facturas', $name, 'public');
            // La URL relativa que guardamos en BD
            $updateData['url_pdf_factura'] = 'facturas/' . $name;
        }

        // Si no tiene permisos para gestionar facturas, no permitir modificar url_pdf_factura
        if (!$canManageInvoices) {
            unset($updateData['url_pdf_factura']);
        }

        $order->update($updateData);

        // If order status changed to 'cancelled', release the inventory
        if ($request->status === 'cancelled' && $oldStatus !== 'cancelled') {
            $orderItems = $order->orderItems()->with('inventory')->get();
            $inventoryIds = $orderItems->pluck('inventory_id')->filter()->toArray();

            if (!empty($inventoryIds)) {
                Inventory::whereIn('id', $inventoryIds)
                    ->whereNotNull('order_id')
                    ->update([
                        'status' => 'disponible',
                        'order_id' => null,
                        'reserved_for_client_id' => null,
                        'updated_at' => now()
                    ]);
            }
        }

        // If order status changed to 'pending' or 'dispatched', reserve/capture the inventory
        if (in_array($request->status, ['pending', 'dispatched']) && in_array($oldStatus, ['draft', 'reserved'], true)) {
            $orderItems = $order->orderItems()->with('inventory')->get();

            foreach ($orderItems as $orderItem) {
                if ($orderItem->inventory) {
                    $orderItem->inventory->update([
                        'status' => $request->status === 'dispatched' ? 'dispatched' : 'reservado',
                        'order_id' => $order->id,
                    ]);
                }
            }
        }

        // Log the action
        AuditLog::logAction(
            'update_order',
            $order,
            Auth::user(),
            $oldData,
            $order->toArray(),
            "Pedido actualizado"
        );

        return redirect()->route('orders.index')
            ->with('success', 'Pedido actualizado exitosamente.');
    }

    /**
     * Cancel the specified order (change status to cancelled instead of deleting).
     */
    public function destroy(Order $order)
    {
        $this->authorizeAction('delete');

        $previousStatus = $order->status;
        $orderData = $order->toArray();

        // Release inventory items
        $orderItems = $order->orderItems()->with('inventory')->get();
        $inventoryIds = $orderItems->pluck('inventory_id')->filter()->toArray();

        $restoredCount = 0;
        if (!empty($inventoryIds)) {
            $restoredCount = Inventory::whereIn('id', $inventoryIds)
                ->whereNotNull('order_id')
                ->update([
                    'status' => 'disponible',
                    'order_id' => null,
                    'reserved_for_client_id' => null,
                    'updated_at' => now()
                ]);
        }

        // Change status to cancelled instead of deleting
        $order->update(['status' => 'cancelled']);

        // Log the action
        AuditLog::logAction(
            'cancel_order',
            $order,
            Auth::user(),
            $orderData,
            ['status' => 'cancelled'],
            "Pedido cancelado (estado anterior: {$previousStatus}) - {$restoredCount} cajas liberadas"
        );

        return redirect()->route('orders.index')
            ->with('success', 'Pedido cancelado exitosamente.');
    }

    /**
     * Get all clients for order creation.
     */
    public function getClients(): JsonResponse
    {
        $clients = Client::active()->orderBy('client_name')->get();

        return response()->json([
            'success' => true,
            'clients' => $clients,
        ]);
    }

    /**
     * Save draft order.
     */
    public function saveDraftOrder(Request $request): JsonResponse
    {
        $request->validate([
            'client_id' => 'required|exists:clients,id',
            'operario_id' => 'nullable|string|max:20',
            'notes' => 'nullable|string',
            'boxes' => 'required|array|min:1',
            'boxes.*' => 'string|max:50',
        ]);

        try {
            DB::beginTransaction();

            // Create order
            $order = Order::create([
                'client_id' => $request->client_id,
                'user_id' => Auth::id(),
                'operario_id' => $request->operario_id,
                'notes' => $request->notes,
                'status' => 'draft',
                'order_date' => now(),
            ]);

            // Get inventory items for each box and RESERVE them (change status to 'reservado')
            foreach ($request->boxes as $boxNumber) {
                $inventoryItems = Inventory::byBox($boxNumber)
                    ->where('status', 'disponible')
                    ->get();

                foreach ($inventoryItems as $item) {
                    // Create OrderItem
                    OrderItem::create([
                        'order_id' => $order->id,
                        'inventory_id' => $item->id,
                        'full_barcode' => $item->full_barcode,
                        'mocaco' => $item->mocaco,
                        'n_carton' => $item->n_carton,
                    ]);

                    // RESERVE the inventory item (set status to 'reservado' and link order_id)
                    $item->update([
                        'status' => 'reservado',
                        'order_id' => $order->id,
                    ]);
                }
            }

            // Log the action
            AuditLog::logAction(
                'create_order',
                $order,
                Auth::user(),
                [],
                $order->toArray(),
                "Borrador de pedido creado con {$order->orderItems()->count()} artículos"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'order_id' => $order->id,
                'message' => 'Borrador guardado con éxito.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error al guardar el borrador: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get draft orders.
     */
    public function getDraftOrders(): JsonResponse
    {
        $orders = Order::draft()
            ->with(['client', 'user'])
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'orders' => $orders,
        ]);
    }

    /**
     * Confirm and dispatch order.
     */
    public function confirmDispatch(Request $request): JsonResponse
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
        ]);

        try {
            DB::beginTransaction();

            $order = Order::findOrFail($request->order_id);

            if (!in_array($order->status, ['draft', 'reserved'], true)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Solo se pueden despachar pedidos en estado borrador o reservado.',
                ], 400);
            }

            // Get all inventory items in this order
            $orderItems = $order->orderItems()->with('inventory')->get();
            $inventoryIds = $orderItems->pluck('inventory_id')->toArray();

            // Update inventory status from 'reservado' to 'dispatched' (not deleted!)
            $updatedCount = Inventory::whereIn('id', $inventoryIds)
                ->where('status', 'reservado')
                ->update([
                    'status' => 'dispatched',
                    'updated_at' => now()
                ]);

            if ($updatedCount !== count($inventoryIds)) {
                throw new \Exception('No se pudieron actualizar todos los artículos del inventario.');
            }

            // Update order status
            $order->dispatch();

            // Log the action
            AuditLog::logAction(
                'dispatch_order',
                $order,
                Auth::user(),
                ['status' => 'draft'],
                ['status' => 'dispatched', 'dispatched_date' => now()],
                "Pedido despachado con {$updatedCount} artículos"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pedido confirmado y enviado con éxito.',
                'dispatched_items' => $updatedCount,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error al confirmar el despacho: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get client orders history.
     */
    public function getClientOrders(Request $request): JsonResponse
    {
        $request->validate([
            'client_id' => 'required|exists:clients,id',
        ]);

        $orders = Order::dispatched()
            ->where('client_id', $request->client_id)
            ->with(['orderItems.inventory'])
            ->orderBy('order_date', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'orders' => $orders,
        ]);
    }

    /**
     * Get order details.
     */
    public function getOrderDetails(Request $request): JsonResponse
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
        ]);

        $order = Order::with([
            'client',
            'user',
            'orderItems.inventory.article'
        ])->findOrFail($request->order_id);

        return response()->json([
            'success' => true,
            'order' => $order,
        ]);
    }

    /**
     * Cancel order.
     */
    public function cancelOrder(Request $request): JsonResponse
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
        ]);

        try {
            DB::beginTransaction();

            $order = Order::findOrFail($request->order_id);

            if ($order->status === 'dispatched') {
                return response()->json([
                    'success' => false,
                    'message' => 'No se puede cancelar un pedido ya despachado.',
                ], 400);
            }

            $oldStatus = $order->status;

            // Get inventory items from this order
            $orderItems = $order->orderItems()->with('inventory')->get();
            $inventoryIds = $orderItems->pluck('inventory_id')->filter()->toArray();

            // RESTORE inventory items to 'disponible' status (regardless of current status)
            $restoredCount = 0;
            if (!empty($inventoryIds)) {
                $restoredCount = Inventory::whereIn('id', $inventoryIds)
                    ->whereNotNull('order_id') // Only release items that are actually linked to an order
                    ->update([
                        'status' => 'disponible',
                        'order_id' => null,
                        'reserved_for_client_id' => null,
                        'updated_at' => now()
                    ]);
            }

            $order->update(['status' => 'cancelled']);

            // Log the action
            AuditLog::logAction(
                'cancel_order',
                $order,
                Auth::user(),
                ['status' => $oldStatus],
                ['status' => 'cancelled'],
                "Pedido cancelado - {$restoredCount} items restaurados al inventario"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pedido cancelado correctamente.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error al cancelar el pedido: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Export orders to Excel format.
     */
    public function export(Request $request)
    {
        $query = Order::with(['client', 'user', 'orderItems.inventory']);

        // Apply same filters as index
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('id', 'like', "%{$search}%")
                    ->orWhereHas('client', function ($clientQuery) use ($search) {
                        $clientQuery->where('client_name', 'like', "%{$search}%");
                    });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('client_id')) {
            $query->where('client_id', $request->client_id);
        }

        $orders = $query->orderBy('created_at', 'desc')->get();

        $filename = 'pedidos_export_' . date('Y-m-d_His') . '.xlsx';

        return Excel::download(new OrdersListExport($orders), $filename);
    }

    /**
     * Export a single order to Excel format with all items.
     */
    public function exportOrder(Order $order)
    {
        $clientName = preg_replace('/[^A-Za-z0-9_-]/', '_', $order->client->client_name ?? 'N/A');
        $filename = 'pedido_' . $order->id . '_cliente_' . $clientName . '_' . date('Y-m-d') . '.xlsx';

        return Excel::download(new OrderDetailExport($order), $filename);
    }

    /**
     * Descargar factura PDF del pedido
     */
    public function downloadInvoice(Order $order)
    {
        $this->authorizeAction('view');

        if (!$order->url_pdf_factura) {
            return redirect()->route('orders.show', $order)
                ->with('error', 'La factura no está disponible aún.');
        }

        // Si es una URL externa, redirigir
        if (filter_var($order->url_pdf_factura, FILTER_VALIDATE_URL)) {
            return redirect($order->url_pdf_factura);
        }

        // Si es una ruta de archivo local, servir el archivo
        $filePath = storage_path('app/public/' . $order->url_pdf_factura);

        if (!file_exists($filePath)) {
            return redirect()->route('orders.show', $order)
                ->with('error', 'El archivo de factura no se encuentra en el servidor.');
        }

        // Obtener el nombre del archivo para la descarga
        $fileName = basename($order->url_pdf_factura);

        return response()->download($filePath, $fileName, [
            'Content-Type' => 'application/pdf',
        ]);
    }

    /**
     * Importar reservas desde Excel y crear pedidos en estado reservado.
     */
    public function importReservations(Request $request)
    {
        $this->authorizeAction('create');

        $request->validate([
            'reservations_file' => 'required|file|mimes:xlsx,xls,csv',
        ]);

        try {
            $import = new OrderReservationsImport(Auth::id());
            Excel::import($import, $request->file('reservations_file'));

            $summary = $import->getSummary();
            $message = "Importación completada: {$summary['reserved_boxes']} cajas reservadas ({$summary['reserved_units']} unidades).";

            return redirect()->route('orders.index')
                ->with('success', $message)
                ->with('import_summary', $summary);
        } catch (\Throwable $e) {
            Log::error('Error al importar reservas de pedidos: ' . $e->getMessage());

            return redirect()->route('orders.index')
                ->with('error', 'Error al importar las reservas: ' . $e->getMessage());
        }
    }

    /**
     * Descargar plantilla de Excel para importación de reservas.
     */
    public function downloadReservationsTemplate()
    {
        $this->authorizeAction('view');

        $filename = 'plantilla_reservas_' . date('Y-m-d_His') . '.xlsx';

        return Excel::download(new OrderReservationTemplateExport(), $filename);
    }
}
