<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Models\AuditLog;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class LocationController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Display the location assignment interface.
     */
    public function index()
    {
        $this->authorizeAction('view');
        
        return view('location.index');
    }

    /**
     * Process location assignment for scanned boxes.
     */
    public function assignLocation(Request $request): JsonResponse
    {
        $this->authorizeAction('assign');
        
        $request->validate([
            'ubicacion' => 'required|string|max:50',
            'operario' => 'required|string|max:50',
            'cajas' => 'required|array',
            'cajas.*' => 'string|max:50',
        ]);

        try {
            DB::beginTransaction();

            $ubicacion = trim($request->ubicacion);
            $operario = trim($request->operario);
            $cajas = $request->cajas;
            $updatedCount = 0;

            foreach ($cajas as $boxNumber) {
                $trimmedBoxNumber = trim($boxNumber);
                
                $updated = Inventory::where('n_carton', $trimmedBoxNumber)
                    ->update([
                        'ubicacion' => $ubicacion,
                        'updated_at' => now(),
                    ]);

                $updatedCount += $updated;
 
                // Log the action for each updated item
                if ($updated > 0) {
                    $items = Inventory::where('n_carton', $trimmedBoxNumber)
                        ->where('ubicacion', $ubicacion)
                        ->get();

                    foreach ($items as $item) {
                        AuditLog::logAction(
                            'assign_location',
                            $item,
                            Auth::user(),
                            [],
                            ['ubicacion' => $ubicacion],
                            "Ubicación $ubicacion asignada por operario $operario a caja $trimmedBoxNumber"
                        );
                    }
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Ubicación $ubicacion asignada exitosamente a $updatedCount items en " . count($cajas) . " caja(s).",
                'updated_count' => $updatedCount,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error al asignar ubicación: ' . $e->getMessage(),
            ], 500);
        }
    }
}
