<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Inventory;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ClientPortalController extends Controller
{
    /**
     * Mostrar lista de reservas/pedidos del cliente autenticado
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        if (!$user->client_id) {
            abort(403, 'Tu cuenta no está asociada a un cliente.');
        }

        $query = Order::where('client_id', $user->client_id)
            ->with(['client', 'user', 'orderItems'])
            ->whereIn('status', ['reservado', 'dispatched', 'confirmed', 'pending']);

        // Filtros opcionales
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('id', 'like', "%{$search}%")
                  ->orWhere('notes', 'like', "%{$search}%");
            });
        }

        $orders = $query->orderBy('order_date', 'desc')
            ->paginate(15)
            ->appends($request->except('page'));

        return view('portal.index', compact('orders'));
    }

    /**
     * Mostrar detalle de una reserva/pedido específico con Packing List agrupado
     */
    public function show(Order $order)
    {
        $user = Auth::user();

        // Verificar que el pedido pertenece al cliente del usuario
        if ($order->client_id !== $user->client_id) {
            abort(403, 'No tienes permiso para ver este pedido.');
        }

        // Cargar relaciones necesarias
        $order->load(['client', 'user', 'orderItems.inventory']);

        // Obtener Packing List agrupado por EAN/FAMILIA
        $packingList = $this->getPackingListGrouped($order);

        return view('portal.show', compact('order', 'packingList'));
    }

    /**
     * Descargar factura PDF
     */
    public function downloadInvoice(Order $order)
    {
        $user = Auth::user();

        // Verificar que el pedido pertenece al cliente del usuario
        if ($order->client_id !== $user->client_id) {
            abort(403, 'No tienes permiso para descargar esta factura.');
        }


        if (!$order->url_pdf_factura) {
            return redirect()->route('portal.show', $order)
                ->with('error', 'La factura no está disponible aún.');
        }

        // Si es una URL externa, redirigir
        if (filter_var($order->url_pdf_factura, FILTER_VALIDATE_URL)) {
            return redirect($order->url_pdf_factura);
        }

        // Si es una ruta de archivo local, servir el archivo
        $filePath = storage_path('app/public/' . $order->url_pdf_factura);
        if (!file_exists($filePath)) {
            return redirect()->route('portal.show', $order)
                ->with('error', 'El archivo de factura no se encuentra en el servidor.');
        }

        // Obtener el nombre del archivo para la descarga
        $fileName = basename($order->url_pdf_factura);
        
        return response()->download($filePath, $fileName, [
            'Content-Type' => 'application/pdf',
        ]);
    }

    /**
     * Descargar pedido en Excel
     */
    public function downloadOrderExcel(Order $order)
    {
        $user = Auth::user();

        // Verificar que el pedido pertenece al cliente del usuario
        if ($order->client_id !== $user->client_id) {
            abort(403, 'No tienes permiso para descargar este pedido.');
        }

        // Cargar relaciones necesarias
        $order->load(['client', 'user', 'orderItems.inventory']);

        // Obtener Packing List agrupado
        $packingList = $this->getPackingListGrouped($order);

        $filename = 'pedido_' . $order->id . '_' . date('Y-m-d_His') . '.xlsx';
        return Excel::download(new \App\Exports\ClientPortalOrderExport($order, $packingList), $filename);
    }

    /**
     * Obtener Packing List agrupado por EAN (mocaco) y FAMILIA
     */
    protected function getPackingListGrouped(Order $order): array
    {
        // Obtener todos los items del pedido con información del inventario
        $orderItems = $order->orderItems()
            ->with('inventory')
            ->get();

        $grouped = [];

        foreach ($orderItems as $item) {
            if (!$item->inventory) {
                continue;
            }

            $inventory = $item->inventory;
            
            // Clave de agrupación: MOCACO + FAMILIA
            $key = ($inventory->mocaco ?? 'SIN_MOCACO') . '|' . ($inventory->familia_articulo_description ?? 'SIN_FAMILIA');
            
            if (!isset($grouped[$key])) {
                $grouped[$key] = [
                    'mocaco' => $inventory->mocaco ?? 'N/A',
                    'familia' => $inventory->familia_articulo_description ?? 'N/A',
                    'seccion' => $inventory->seccion ?? null,
                    'cadena' => $inventory->cadena ?? null,
                    'campana' => $inventory->campana ?? null,
                    'conteneur' => $inventory->conteneur ?? null,
                    'mercado_origen_articulo' => $inventory->mercado_origen_articulo ?? null,
                    'precio_pvp_maximo_temporada' => $inventory->precio_pvp_maximo_temporada ?? null,
                    'partida_arancelaria' => $inventory->partida_arancelaria ?? null,
                    'composition' => $inventory->composition ?? null,
                    'peso_unitario' => $inventory->peso_unitario ?? null,
                    'grupo_arancelario' => $inventory->grupo_arancelario ?? null,
                    'season_int' => $inventory->season_int ?? null,
                    'categoria_seleccionada' => $inventory->categoria_seleccionada ?? null,
                    'famillie_usuario' => $inventory->famillie_usuario ?? null,
                    'detail_usuario' => $inventory->detail_usuario ?? null,
                    'unidades' => 0,
                    'cajas' => [],
                ];
            }

            // Incrementar unidades
            $grouped[$key]['unidades']++;
            
            // Agregar caja única si no existe
            $nCarton = $inventory->n_carton ?? 'N/A';
            if (!in_array($nCarton, $grouped[$key]['cajas'])) {
                $grouped[$key]['cajas'][] = $nCarton;
            }
        }

        // Convertir a array indexado y ordenar por familia
        return array_values($grouped);
    }
}

