<?php

namespace App\Http\Controllers;

use App\Models\Client;
use App\Models\Order;
use App\Models\OrderItem;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ClientHistoryController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Display the client history form.
     */
    public function index()
    {
        $this->authorizeAction('view');

        return view('client-history.index');
    }

    /**
     * Get clients for Select2 dropdown.
     */
    public function getClients()
    {
        $this->authorizeAction('view');

        $clients = Client::select('id', 'client_name', 'email', 'phone', 'address')
            ->where('is_active', true)
            ->orderBy('client_name')
            ->get();

        return response()->json($clients);
    }

    /**
     * Get client order history.
     */
    public function getClientOrders($clientId)
    {
        $this->authorizeAction('view');

        try {
            \Log::info('getClientOrders - Request for client: ' . $clientId);

            $orders = Order::with(['orderItems'])
                ->where('client_id', $clientId)
                ->where('status', 'dispatched')
                ->orderBy('created_at', 'desc')
                ->get();

            \Log::info('getClientOrders - Found orders: ' . $orders->count());

            $formattedOrders = $orders->map(function ($order) {
                return [
                    'id' => $order->id,
                    'order_date' => $order->order_date ? $order->order_date->toDateTimeString() : $order->created_at->toDateTimeString(),
                    'dispatched_date' => $order->dispatched_date ? $order->dispatched_date->toDateTimeString() : null,
                    'operario_id' => $order->operario_id ?? 'N/A',
                    'notes' => $order->notes ?? '',
                    'order_items' => $order->orderItems->map(function ($item) {
                        return [
                            'full_barcode' => $item->full_barcode,
                            'mocaco' => $item->mocaco ?? 'N/A',
                            'n_carton' => $item->n_carton ?? 'N/A',
                        ];
                    })
                ];
            });

            \Log::info('getClientOrders - Returning: ' . json_encode($formattedOrders));

            return response()->json($formattedOrders);
        } catch (\Exception $e) {
            \Log::error('getClientOrders - Error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get order details with items.
     */
    public function getOrderDetails($orderId)
    {
        $this->authorizeAction('view');

        $order = Order::with(['client', 'orderItems'])
            ->findOrFail($orderId);

        return response()->json([
            'id' => $order->id,
            'order_date' => $order->order_date->toDateTimeString(),
            'dispatched_date' => $order->dispatched_date ? $order->dispatched_date->toDateTimeString() : null,
            'operario_id' => $order->operario_id ?? 'N/A',
            'notes' => $order->notes ?? '',
            'client' => [
                'id' => $order->client->id,
                'client_name' => $order->client->client_name,
                'email' => $order->client->email ?? 'N/A',
                'phone' => $order->client->phone ?? 'N/A',
            ],
            'order_items' => $order->orderItems->map(function ($item) {
                return [
                    'full_barcode' => $item->full_barcode,
                    'mocaco' => $item->mocaco ?? 'N/A',
                    'n_carton' => $item->n_carton ?? 'N/A',
                ];
            })
        ]);
    }

    /**
     * Get client statistics.
     */
    public function getClientStats($clientId)
    {
        $this->authorizeAction('view');

        $client = Client::findOrFail($clientId);

        $stats = [
            'total_orders' => Order::where('client_id', $clientId)->where('status', 'dispatched')->count(),
            'total_items' => OrderItem::whereHas('order', function ($query) use ($clientId) {
                $query->where('client_id', $clientId)->where('status', 'dispatched');
            })->count(),
            'total_boxes' => OrderItem::whereHas('order', function ($query) use ($clientId) {
                $query->where('client_id', $clientId)->where('status', 'dispatched');
            })->distinct('n_carton')->count(),
            'last_order_date' => Order::where('client_id', $clientId)
                ->where('status', 'dispatched')
                ->orderBy('dispatched_date', 'desc')
                ->value('dispatched_date'),
            'client_info' => $client
        ];

        return response()->json($stats);
    }

    /**
     * Export client history to CSV.
     */
    /**
     * Export client history to Excel.
     */
    public function exportClientHistory($clientId)
    {
        $this->authorizeAction('export');

        $client = Client::findOrFail($clientId);
        $orders = Order::with(['orderItems'])
            ->where('client_id', $clientId)
            ->where('status', 'dispatched')
            ->orderBy('order_date', 'desc')
            ->get();

        $filename = "historial_cliente_{$client->client_name}_{$clientId}_" . now()->format('Y-m-d') . '.xlsx';

        return \Maatwebsite\Excel\Facades\Excel::download(new \App\Exports\ClientHistoryExport($orders), $filename);
    }
}
