<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Models\Article;
use App\Models\AuditLog;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ArticleReviewController extends Controller
{
    use AuthorizesResourceAccess;

    /**
     * Display a listing of articles that need review.
     */
    public function index(Request $request)
    {
        $this->authorizeAction('view');

        $query = Inventory::where('needs_review', true)
            ->with(['article']);

        // Filtros
        if ($request->filled('conteneur')) {
            $query->where('conteneur', 'like', '%' . $request->conteneur . '%');
        }

        if ($request->filled('mocaco')) {
            $query->where('mocaco', 'like', '%' . $request->mocaco . '%');
        }

        if ($request->filled('n_carton')) {
            $query->where('n_carton', 'like', '%' . $request->n_carton . '%');
        }

        if ($request->filled('date_from')) {
            $query->whereDate('fecha_escaneo', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('fecha_escaneo', '<=', $request->date_to);
        }

        $perPage = $request->get('per_page', 20);
        $articlesNeedingReview = $query->orderBy('fecha_escaneo', 'desc')->paginate($perPage)->appends($request->except('page'));

        return view('article-review.index', compact('articlesNeedingReview'));
    }

    /**
     * Get suggestions for an article based on mocaco.
     */
    public function getSuggestions(Inventory $inventory)
    {
        $this->authorizeAction('view');

        try {
            $suggestions = $this->generateSuggestions($inventory->mocaco);

            return response()->json([
                'success' => true,
                'suggestions' => $suggestions,
                'inventory' => $inventory,
            ]);
        } catch (\Exception $e) {
            Log::error('Error al obtener sugerencias: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al obtener sugerencias: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update article with confirmed data.
     */
    public function updateArticle(Request $request, Inventory $inventory)
    {
        $this->authorizeAction('update');

        $request->validate([
            'seccion' => 'nullable|string|max:100',
            'familia' => 'nullable|string|max:255',
            'cadena' => 'nullable|string|max:100',
            'mercado_origen_articulo' => 'nullable|string|max:100',
            'precio_pvp_maximo_temporada' => 'nullable|numeric',
            'partida_arancelaria' => 'nullable|string|max:100',
            'composition' => 'nullable|string',
            'campana' => 'nullable|string|max:100',
            'peso_unitario' => 'nullable|numeric',
            'grupo_arancelario' => 'nullable|string|max:100',
            'review_notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $oldData = $inventory->toArray();

            // Actualizar el inventario
            $inventory->update([
                'seccion' => $request->seccion,
                'familia_articulo_description' => $request->familia,
                'cadena' => $request->cadena,
                'mercado_origen_articulo' => $request->mercado_origen_articulo,
                'precio_pvp_maximo_temporada' => $request->precio_pvp_maximo_temporada,
                'partida_arancelaria' => $request->partida_arancelaria,
                'composition' => $request->composition,
                'campana' => $request->campana,
                'peso_unitario' => $request->peso_unitario,
                'grupo_arancelario' => $request->grupo_arancelario,
                'needs_review' => false,
                'reviewed_at' => now(),
                'reviewed_by' => Auth::id(),
                'review_notes' => $request->review_notes,
            ]);

            // Actualizar o crear el artículo
            $article = Article::where('mocaco', $inventory->mocaco)
                ->where(function ($query) use ($inventory) {
                    if ($inventory->campana) {
                        $query->where('campana', $inventory->campana);
                    } else {
                        $query->whereNull('campana');
                    }
                })
                ->first();

            if ($article) {
                $article->update([
                    'seccion' => $request->seccion,
                    'familia' => $request->familia,
                    'cadena' => $request->cadena,
                    'mercado_origen_articulo' => $request->mercado_origen_articulo,
                    'precio_pvp_maximo_temporada' => $request->precio_pvp_maximo_temporada,
                    'partida_arancelaria' => $request->partida_arancelaria,
                    'composition' => $request->composition,
                    'peso_unitario' => $request->peso_unitario,
                    'grupo_arancelario' => $request->grupo_arancelario,
                ]);
            } else {
                Article::create([
                    'mocaco' => $inventory->mocaco,
                    'seccion' => $request->seccion,
                    'familia' => $request->familia,
                    'cadena' => $request->cadena,
                    'mercado_origen_articulo' => $request->mercado_origen_articulo,
                    'precio_pvp_maximo_temporada' => $request->precio_pvp_maximo_temporada,
                    'partida_arancelaria' => $request->partida_arancelaria,
                    'composition' => $request->composition,
                    'campana' => $request->campana,
                    'peso_unitario' => $request->peso_unitario,
                    'grupo_arancelario' => $request->grupo_arancelario,
                ]);
            }

            // Log the action
            AuditLog::logAction(
                'review_article',
                $inventory,
                Auth::user(),
                $oldData,
                $inventory->toArray(),
                "Artículo revisado y actualizado: {$inventory->mocaco}"
            );

            DB::commit();

            return redirect()->route('article-review.index')
                ->with('success', 'Artículo revisado y actualizado correctamente.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al actualizar artículo: ' . $e->getMessage());

            return back()->withInput()
                ->with('error', 'Error al actualizar: ' . $e->getMessage());
        }
    }

    /**
     * Bulk update articles with same mocaco.
     */
    public function bulkUpdate(Request $request)
    {
        $this->authorizeAction('update');

        $request->validate([
            'mocaco' => 'required|string|max:20',
            'seccion' => 'nullable|string|max:100',
            'familia' => 'nullable|string|max:255',
            'cadena' => 'nullable|string|max:100',
            'mercado_origen_articulo' => 'nullable|string|max:100',
            'precio_pvp_maximo_temporada' => 'nullable|numeric',
            'partida_arancelaria' => 'nullable|string|max:100',
            'composition' => 'nullable|string',
            'campana' => 'nullable|string|max:100',
            'peso_unitario' => 'nullable|numeric',
            'grupo_arancelario' => 'nullable|string|max:100',
        ]);

        try {
            DB::beginTransaction();

            // Actualizar todos los inventarios con el mismo mocaco que necesitan revisión
            $updated = Inventory::where('mocaco', $request->mocaco)
                ->where('needs_review', true)
                ->update([
                    'seccion' => $request->seccion,
                    'familia_articulo_description' => $request->familia,
                    'cadena' => $request->cadena,
                    'mercado_origen_articulo' => $request->mercado_origen_articulo,
                    'precio_pvp_maximo_temporada' => $request->precio_pvp_maximo_temporada,
                    'partida_arancelaria' => $request->partida_arancelaria,
                    'composition' => $request->composition,
                    'campana' => $request->campana,
                    'peso_unitario' => $request->peso_unitario,
                    'grupo_arancelario' => $request->grupo_arancelario,
                    'needs_review' => false,
                    'reviewed_at' => now(),
                    'reviewed_by' => Auth::id(),
                ]);

            // Actualizar o crear el artículo
            Article::updateOrCreate(
                [
                    'mocaco' => $request->mocaco,
                    'campana' => $request->campana,
                ],
                [
                    'seccion' => $request->seccion,
                    'familia' => $request->familia,
                    'cadena' => $request->cadena,
                    'mercado_origen_articulo' => $request->mercado_origen_articulo,
                    'precio_pvp_maximo_temporada' => $request->precio_pvp_maximo_temporada,
                    'partida_arancelaria' => $request->partida_arancelaria,
                    'composition' => $request->composition,
                    'peso_unitario' => $request->peso_unitario,
                    'grupo_arancelario' => $request->grupo_arancelario,
                ]
            );

            DB::commit();

            return redirect()->route('article-review.index')
                ->with('success', "{$updated} artículos actualizados correctamente.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error en actualización masiva: ' . $e->getMessage());

            return back()->withInput()
                ->with('error', 'Error en actualización masiva: ' . $e->getMessage());
        }
    }

    /**
     * Generate suggestions based on historical data.
     */
    protected function generateSuggestions($mocaco)
    {
        $suggestions = [];

        // 1. Buscar artículos con el mismo mocaco que tienen información completa
        $articlesWithSameMocaco = Article::where('mocaco', $mocaco)
            ->whereNotNull('seccion')
            ->whereNotNull('familia')
            ->get();

        foreach ($articlesWithSameMocaco as $article) {
            $suggestions[] = [
                'source' => 'article_master',
                'confidence' => 'high',
                'data' => [
                    'seccion' => $article->seccion,
                    'familia' => $article->familia,
                    'cadena' => $article->cadena,
                    'mercado_origen_articulo' => $article->mercado_origen_articulo,
                    'precio_pvp_maximo_temporada' => $article->precio_pvp_maximo_temporada,
                    'partida_arancelaria' => $article->partida_arancelaria,
                    'composition' => $article->composition,
                    'campana' => $article->campana,
                    'peso_unitario' => $article->peso_unitario,
                    'grupo_arancelario' => $article->grupo_arancelario,
                ],
            ];
        }

        // 2. Buscar en inventario histórico con el mismo mocaco
        $inventoryWithSameMocaco = Inventory::where('mocaco', $mocaco)
            ->where('needs_review', false)
            ->whereNotNull('seccion')
            ->whereNotNull('familia_articulo_description')
            ->select(
                'seccion',
                'familia_articulo_description',
                'cadena',
                'mercado_origen_articulo',
                'precio_pvp_maximo_temporada',
                'partida_arancelaria',
                'composition',
                'campana',
                'peso_unitario',
                'grupo_arancelario'
            )
            ->distinct()
            ->get();

        foreach ($inventoryWithSameMocaco as $inv) {
            // Evitar duplicados
            $exists = collect($suggestions)->contains(function ($s) use ($inv) {
                return $s['data']['seccion'] === $inv->seccion &&
                    $s['data']['familia'] === $inv->familia_articulo_description;
            });

            if (!$exists) {
                $suggestions[] = [
                    'source' => 'inventory_history',
                    'confidence' => 'medium',
                    'data' => [
                        'seccion' => $inv->seccion,
                        'familia' => $inv->familia_articulo_description,
                        'cadena' => $inv->cadena,
                        'mercado_origen_articulo' => $inv->mercado_origen_articulo,
                        'precio_pvp_maximo_temporada' => $inv->precio_pvp_maximo_temporada,
                        'partida_arancelaria' => $inv->partida_arancelaria,
                        'composition' => $inv->composition,
                        'campana' => $inv->campana,
                        'peso_unitario' => $inv->peso_unitario,
                        'grupo_arancelario' => $inv->grupo_arancelario,
                    ],
                ];
            }
        }

        return $suggestions;
    }
}
