<?php

namespace App\Exports;

use App\Models\Inventory;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class ProductivityExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $filters;

    public function __construct(array $filters)
    {
        $this->filters = $filters;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $query = Inventory::select(
            'n_id_operario',
            DB::raw('DATE(fecha_escaneo) as fecha'),
            DB::raw('COUNT(*) as unidades_escaneadas'),
            DB::raw('COUNT(DISTINCT n_carton) as cajas_procesadas')
        )
            ->groupBy('n_id_operario', DB::raw('DATE(fecha_escaneo)'))
            ->orderBy('fecha', 'desc')
            ->orderBy('n_id_operario');

        // Aplicar filtros
        if (isset($this->filters['operario_id']) && $this->filters['operario_id']) {
            $query->where('n_id_operario', $this->filters['operario_id']);
        }

        if (isset($this->filters['fecha_desde']) && $this->filters['fecha_desde']) {
            $query->whereDate('fecha_escaneo', '>=', $this->filters['fecha_desde']);
        }

        if (isset($this->filters['fecha_hasta']) && $this->filters['fecha_hasta']) {
            $query->whereDate('fecha_escaneo', '<=', $this->filters['fecha_hasta']);
        }

        return $query->get();
    }

    public function headings(): array
    {
        return [
            'Operario',
            'Fecha',
            'Unidades Escaneadas',
            'Cajas Procesadas',
            'Promedio por Caja'
        ];
    }

    public function map($row): array
    {
        $fecha = \Carbon\Carbon::parse($row->fecha)->format('d/m/Y');
        $promedio = $row->cajas_procesadas > 0 ? round($row->unidades_escaneadas / $row->cajas_procesadas, 2) : 0;

        return [
            $row->n_id_operario,
            $fecha,
            $row->unidades_escaneadas,
            $row->cajas_procesadas,
            $promedio
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
}
