<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use App\Models\Order;

class OrdersListExport implements FromArray, WithHeadings, WithMapping, WithTitle, WithColumnWidths, WithEvents
{
    protected $orders;

    public function __construct($orders)
    {
        $this->orders = $orders;
    }

    /**
     * @return array
     */
    public function array(): array
    {
        $items = [];
        foreach ($this->orders as $order) {
            foreach ($order->orderItems as $orderItem) {
                $items[] = [
                    'order' => $order,
                    'orderItem' => $orderItem
                ];
            }
        }
        return $items;
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'ID Pedido',
            'Cliente',
            'Email Cliente',
            'Teléfono',
            'Operario Pedido',
            'Usuario',
            'Status Pedido',
            'Fecha Pedido',
            'Fecha Despacho',
            'Notas Pedido',
            'Código de Barras',
            'MOCACO',
            'Nº Carton',
            'Ubicación',
            'Contenedor',
            'Operario Item',
            'Season INT',
            'Categoría',
            'Familia Usuario',
            'Detail Usuario',
            'Sección',
            'Familia Artículo',
            'Cadena',
            'Mercado Origen',
            'Precio PVP Max',
            'Partida Arancelaria',
            'Composition',
            'Campaña',
            'Peso Unitario',
            'Grupo Arancelario',
            'Status Item',
            'Order ID Item',
        ];
    }

    /**
     * @param array $row
     * @return array
     */
    public function map($row): array
    {
        $order = $row['order'];
        $orderItem = $row['orderItem'];
        $inventory = $orderItem->inventory;
        
        return [
            // Información del Pedido
            $order->id,
            $order->client->client_name ?? '',
            $order->client->email ?? '',
            $order->client->phone ?? '',
            $order->operario_id ?? '',
            $order->user->name ?? '',
            ucfirst($order->status),
            $order->order_date ? $order->order_date->format('d/m/Y H:i') : '',
            $order->dispatched_date ? $order->dispatched_date->format('d/m/Y H:i') : '',
            $order->notes ?? '',
            // Información del Item de Inventario
            $inventory ? ($inventory->full_barcode ?? $orderItem->full_barcode ?? '') : ($orderItem->full_barcode ?? ''),
            $inventory ? ($inventory->mocaco ?? $orderItem->mocaco ?? '') : ($orderItem->mocaco ?? ''),
            $inventory ? ($inventory->n_carton ?? $orderItem->n_carton ?? '') : ($orderItem->n_carton ?? ''),
            $inventory ? $inventory->ubicacion ?? '' : '',
            $inventory ? $inventory->conteneur ?? '' : '',
            $inventory ? $inventory->n_id_operario ?? '' : '',
            $inventory ? $inventory->season_int ?? '' : '',
            $inventory ? $inventory->categoria_seleccionada ?? '' : '',
            $inventory ? $inventory->famillie_usuario ?? '' : '',
            $inventory ? $inventory->detail_usuario ?? '' : '',
            $inventory ? $inventory->seccion ?? '' : '',
            $inventory ? $inventory->familia_articulo_description ?? '' : '',
            $inventory ? $inventory->cadena ?? '' : '',
            $inventory ? $inventory->mercado_origen_articulo ?? '' : '',
            $inventory ? $inventory->precio_pvp_maximo_temporada ?? '' : '',
            $inventory ? $inventory->partida_arancelaria ?? '' : '',
            $inventory ? $inventory->composition ?? '' : '',
            $inventory ? $inventory->campana ?? '' : '',
            $inventory ? $inventory->peso_unitario ?? '' : '',
            $inventory ? $inventory->grupo_arancelario ?? '' : '',
            $inventory ? ($inventory->status ?? 'disponible') : '',
            $inventory ? $inventory->order_id ?? '' : '',
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return 'Pedidos';
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                
                // Estilo para los headers de la tabla
                $sheet->getStyle('A1:AF1')->getFont()->setBold(true)->setSize(12);
                $sheet->getStyle('A1:AF1')->getFill()
                    ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                    ->setStartColor(new \PhpOffice\PhpSpreadsheet\Style\Color(\PhpOffice\PhpSpreadsheet\Style\Color::COLOR_YELLOW));
            },
        ];
    }

    /**
     * @return array
     */
    public function columnWidths(): array
    {
        return [
            'A' => 12, // ID Pedido
            'B' => 25, // Cliente
            'C' => 25, // Email Cliente
            'D' => 15, // Teléfono
            'E' => 12, // Operario
            'F' => 15, // Usuario
            'G' => 12, // Status
            'H' => 18, // Fecha Pedido
            'I' => 18, // Fecha Despacho
            'J' => 40, // Notas
            'K' => 20, // Código de Barras
            'L' => 15, // MOCACO
            'M' => 12, // Nº Carton
            'N' => 15, // Ubicación
            'O' => 15, // Contenedor
            'P' => 12, // Operario
            'Q' => 12, // Season INT
            'R' => 15, // Categoría
            'S' => 20, // Familia Usuario
            'T' => 20, // Detail Usuario
            'U' => 12, // Sección
            'V' => 20, // Familia Artículo
            'W' => 15, // Cadena
            'X' => 18, // Mercado Origen
            'Y' => 15, // Precio PVP Max
            'Z' => 18, // Partida Arancelaria
            'AA' => 20, // Composition
            'AB' => 15, // Campaña
            'AC' => 15, // Peso Unitario
            'AD' => 18, // Grupo Arancelario
            'AE' => 12, // Status
            'AF' => 12, // Order ID
        ];
    }
}

