<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Events\AfterSheet;
use App\Models\Inventory;
use Illuminate\Database\Eloquent\Builder;

class InventoryExport implements FromQuery, WithHeadings, WithMapping, WithTitle, WithColumnWidths, WithEvents, WithChunkReading
{
    protected $query;

    public function __construct(Builder $query)
    {
        $this->query = $query;
    }

    /**
     * @return Builder
     */
    public function query()
    {
        return $this->query;
    }

    /**
     * Tamaño del chunk para procesar en lotes
     *
     * @return int
     */
    public function chunkSize(): int
    {
        return 1000; // Procesar 1000 registros a la vez
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'ID',
            'Código de Barras',
            'MOCACO',
            'Nº Carton',
            'Ubicación',
            'Contenedor',
            'Operario',
            'Season INT',
            'Categoría',
            'Familia Usuario',
            'Detail Usuario',
            'Sección',
            'Familia Artículo',
            'Cadena',
            'Mercado Origen',
            'Precio PVP Max',
            'Partida Arancelaria',
            'Composition',
            'Campaña',
            'Peso Unitario',
            'Grupo Arancelario',
            'Talla',
            'Color',
            'Status',
            'Order ID',
            'Notas',
            'Fecha Escaneo',
            'Fecha Creación',
            'Última Actualización',
        ];
    }

    /**
     * @param Inventory $item
     * @return array
     */
    public function map($item): array
    {
        return [
            $item->id,
            $item->full_barcode,
            $item->mocaco,
            $item->n_carton,
            $item->ubicacion ?? '',
            $item->conteneur ?? '',
            $item->n_id_operario ?? '',
            $item->season_int ?? '',
            $item->categoria_seleccionada ?? '',
            $item->famillie_usuario ?? '',
            $item->detail_usuario ?? '',
            $item->seccion ?? '',
            $item->familia_articulo_description ?? '',
            $item->cadena ?? '',
            $item->mercado_origen_articulo ?? '',
            $item->precio_pvp_maximo_temporada ?? '',
            $item->partida_arancelaria ?? '',
            $item->composition ?? '',
            $item->campana ?? '',
            $item->peso_unitario ?? '',
            $item->grupo_arancelario ?? '',
            $item->size ?? '',
            $item->color ?? '',
            $item->status ?? 'disponible',
            $item->order_id ?? '',
            $item->notes ?? '',
            $item->fecha_escaneo ? $item->fecha_escaneo->format('d/m/Y H:i:s') : '',
            $item->created_at ? $item->created_at->format('d/m/Y H:i:s') : '',
            $item->updated_at ? $item->updated_at->format('d/m/Y H:i:s') : '',
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return 'Inventario';
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();

                // Estilo para los headers de la tabla
                $sheet->getStyle('A1:AC1')->getFont()->setBold(true)->setSize(12);
                $sheet->getStyle('A1:AC1')->getFill()
                    ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                    ->setStartColor(new \PhpOffice\PhpSpreadsheet\Style\Color(\PhpOffice\PhpSpreadsheet\Style\Color::COLOR_YELLOW));
            },
        ];
    }

    /**
     * @return array
     */
    public function columnWidths(): array
    {
        return [
            'A' => 8,  // ID
            'B' => 20, // Código de Barras
            'C' => 15, // MOCACO
            'D' => 12, // Nº Carton
            'E' => 15, // Ubicación
            'F' => 15, // Contenedor
            'G' => 12, // Operario
            'H' => 12, // Season INT
            'I' => 15, // Categoría
            'J' => 20, // Familia Usuario
            'K' => 20, // Detail Usuario
            'L' => 12, // Sección
            'M' => 20, // Familia Artículo
            'N' => 15, // Cadena
            'O' => 18, // Mercado Origen
            'P' => 15, // Precio PVP Max
            'Q' => 18, // Partida Arancelaria
            'R' => 20, // Composition
            'S' => 15, // Campaña
            'T' => 15, // Peso Unitario
            'U' => 18, // Grupo Arancelario
            'V' => 10, // Talla
            'W' => 15, // Color
            'X' => 12, // Status
            'Y' => 12, // Order ID
            'Z' => 30, // Notas
            'AA' => 18, // Fecha Escaneo
            'AB' => 18, // Fecha Creación
            'AC' => 18, // Última Actualización
        ];
    }
}
