<?php

namespace App\Exports;

use App\Models\ContainerEntry;
use App\Services\ContainerEntryService;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;

class DiscrepancyReportExport implements FromCollection, WithHeadings, WithMapping, WithTitle, WithColumnWidths, WithStyles, WithEvents
{
    protected $report;
    protected $containerEntry;

    public function __construct(ContainerEntry $containerEntry)
    {
        $this->containerEntry = $containerEntry;
        $service = app(ContainerEntryService::class);
        $this->report = $service->generateDiscrepancyReport($containerEntry);
    }

    public function collection(): Collection
    {
        return collect($this->report['discrepancies'] ?? []);
    }

    public function headings(): array
    {
        return [
            'MOCACO',
            'Unidades Teóricas',
            'Unidades Escaneadas',
            'Diferencia',
            'Tipo',
        ];
    }

    public function map($discrepancy): array
    {
        $tipo = match($discrepancy['tipo']) {
            'faltante' => 'Faltante',
            'exceso' => 'Exceso',
            'no_esperado' => 'No Esperado',
            default => $discrepancy['tipo'],
        };

        return [
            $discrepancy['mocaco'],
            $discrepancy['unidades_teoricas'],
            $discrepancy['unidades_escaneadas'],
            $discrepancy['diferencia'],
            $tipo,
        ];
    }

    public function title(): string
    {
        return 'Discrepancias';
    }

    public function columnWidths(): array
    {
        return [
            'A' => 20,  // MOCACO
            'B' => 18,  // Unidades Teóricas
            'C' => 20,  // Unidades Escaneadas
            'D' => 15,  // Diferencia
            'E' => 15,  // Tipo
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                
                // Agregar información del contenedor y resumen al inicio
                // Insertamos 11 filas para que los encabezados queden en la fila 12
                $sheet->insertNewRowBefore(1, 11);
                
                // Información del contenedor
                $sheet->setCellValue('A1', 'Contenedor:');
                $sheet->setCellValue('B1', $this->containerEntry->n_camion);
                $sheet->setCellValue('A2', 'ID Contenedor:');
                $sheet->setCellValue('B2', $this->containerEntry->id);
                $sheet->setCellValue('A3', 'Fecha Reporte:');
                $sheet->setCellValue('B3', now()->format('Y-m-d H:i:s'));
                
                // Resumen
                $summary = $this->report['summary'] ?? [];
                $sheet->setCellValue('A5', 'RESUMEN');
                $sheet->setCellValue('A6', 'Total Teórico:');
                $sheet->setCellValue('B6', $summary['total_teorico'] ?? 0);
                $sheet->setCellValue('A7', 'Total Escaneado:');
                $sheet->setCellValue('B7', $summary['total_escaneado'] ?? 0);
                $sheet->setCellValue('A8', 'Faltantes:');
                $sheet->setCellValue('B8', abs($summary['total_faltantes'] ?? 0));
                $sheet->setCellValue('A9', 'Excesos:');
                $sheet->setCellValue('B9', $summary['total_excesos'] ?? 0);
                $sheet->setCellValue('A10', 'No Esperados:');
                $sheet->setCellValue('B10', $summary['total_no_esperados'] ?? 0);
                $sheet->setCellValue('A11', 'Total Discrepancias:');
                $sheet->setCellValue('B11', $this->report['total_discrepancias'] ?? 0);
                
                // Estilos para el encabezado
                $sheet->getStyle('A5')->applyFromArray([
                    'font' => ['bold' => true, 'size' => 12],
                ]);
                
                // Estilos para los encabezados de la tabla (fila 12 después de insertar 11 filas)
                $sheet->getStyle('A12:E12')->applyFromArray([
                    'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => '4472C4']
                    ],
                    'alignment' => [
                        'horizontal' => Alignment::HORIZONTAL_CENTER,
                        'vertical' => Alignment::VERTICAL_CENTER,
                    ],
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => Border::BORDER_THIN,
                        ],
                    ],
                ]);
                
                // Aplicar bordes a todas las celdas de datos
                $lastRow = $sheet->getHighestRow();
                if ($lastRow > 12) {
                    $sheet->getStyle('A12:E' . $lastRow)->applyFromArray([
                        'borders' => [
                            'allBorders' => [
                                'borderStyle' => Border::BORDER_THIN,
                            ],
                        ],
                    ]);
                    
                    // Colorear filas según el tipo
                    for ($row = 13; $row <= $lastRow; $row++) {
                        $tipo = $sheet->getCell('E' . $row)->getValue();
                        if ($tipo === 'Faltante') {
                            $sheet->getStyle('A' . $row . ':E' . $row)->applyFromArray([
                                'fill' => [
                                    'fillType' => Fill::FILL_SOLID,
                                    'startColor' => ['rgb' => 'FFE6E6']
                                ],
                            ]);
                        } elseif ($tipo === 'Exceso') {
                            $sheet->getStyle('A' . $row . ':E' . $row)->applyFromArray([
                                'fill' => [
                                    'fillType' => Fill::FILL_SOLID,
                                    'startColor' => ['rgb' => 'FFF4E6']
                                ],
                            ]);
                        } elseif ($tipo === 'No Esperado') {
                            $sheet->getStyle('A' . $row . ':E' . $row)->applyFromArray([
                                'fill' => [
                                    'fillType' => Fill::FILL_SOLID,
                                    'startColor' => ['rgb' => 'E6F3FF']
                                ],
                            ]);
                        }
                    }
                }
                
                // Ajustar altura de filas
                $sheet->getRowDimension(5)->setRowHeight(20);
                $sheet->getRowDimension(12)->setRowHeight(25);
            },
        ];
    }
}

