<?php

namespace App\Exports;

use App\Models\Order;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;

class ClientPortalOrderExport implements FromCollection, WithHeadings, WithMapping, WithTitle, WithColumnWidths, WithStyles, WithEvents
{
    protected $order;
    protected $packingList;

    public function __construct(Order $order, array $packingList)
    {
        $this->order = $order->load(['client', 'user']);
        $this->packingList = $packingList;
    }

    public function collection(): Collection
    {
        return collect($this->packingList);
    }

    public function headings(): array
    {
        return [
            'MOCACO (EAN)',
            'Familia',
            'Sección',
            'Campaña',
            'Unidades',
            'Cajas',
            'Contenedor',
            'Familia Artículo',
            'Cadena',
            'Mercado Origen',
            'Precio PVP Max Temporada',
            'Partida Arancelaria',
            'Composition',
            'Peso Unitario',
            'Grupo Arancelario',
            'Season INT',
            'Categoría Seleccionada',
            'Familia Usuario',
            'Detail Usuario',
        ];
    }

    public function map($item): array
    {
        return [
            $item['mocaco'],
            $item['familia'],
            $item['seccion'] ?? '-',
            $item['campana'] ?? '-',
            $item['unidades'],
            implode(', ', $item['cajas']),
            $item['conteneur'] ?? '-',
            $item['familia'] ?? '-',
            $item['cadena'] ?? '-',
            $item['mercado_origen_articulo'] ?? '-',
            $item['precio_pvp_maximo_temporada'] ?? '-',
            $item['partida_arancelaria'] ?? '-',
            $item['composition'] ?? '-',
            $item['peso_unitario'] ?? '-',
            $item['grupo_arancelario'] ?? '-',
            $item['season_int'] ?? '-',
            $item['categoria_seleccionada'] ?? '-',
            $item['famillie_usuario'] ?? '-',
            $item['detail_usuario'] ?? '-',
        ];
    }

    public function title(): string
    {
        return 'Pedido #' . $this->order->id;
    }

    public function columnWidths(): array
    {
        return [
            'A' => 20,  // MOCACO
            'B' => 25,  // Familia
            'C' => 15,  // Sección
            'D' => 15,  // Campaña
            'E' => 12,  // Unidades
            'F' => 50,  // Cajas
            'G' => 15,  // Contenedor
            'H' => 25,  // Familia Artículo
            'I' => 15,  // Cadena
            'J' => 20,  // Mercado Origen
            'K' => 20,  // Precio PVP Max Temporada
            'L' => 20,  // Partida Arancelaria
            'M' => 30,  // Composition
            'N' => 15,  // Peso Unitario
            'O' => 20,  // Grupo Arancelario
            'P' => 15,  // Season INT
            'Q' => 20,  // Categoría Seleccionada
            'R' => 20,  // Familia Usuario
            'S' => 25,  // Detail Usuario
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                
                // Agregar información del pedido al inicio
                $sheet->insertNewRowBefore(1, 10);
                
                // Información del pedido
                $sheet->setCellValue('A1', 'INFORMACIÓN DEL PEDIDO');
                $sheet->setCellValue('A2', 'ID Pedido:');
                $sheet->setCellValue('B2', $this->order->id);
                $sheet->setCellValue('A3', 'Cliente:');
                $sheet->setCellValue('B3', $this->order->client->client_name ?? 'N/A');
                $sheet->setCellValue('A4', 'Estado:');
                $sheet->setCellValue('B4', ucfirst($this->order->status));
                $sheet->setCellValue('A5', 'Fecha Pedido:');
                $sheet->setCellValue('B5', $this->order->order_date ? $this->order->order_date->format('d/m/Y H:i') : '');
                if ($this->order->dispatched_date) {
                    $sheet->setCellValue('A6', 'Fecha Despacho:');
                    $sheet->setCellValue('B6', $this->order->dispatched_date->format('d/m/Y H:i'));
                }
                $sheet->setCellValue('A7', 'Total Artículos:');
                $sheet->setCellValue('B7', $this->order->orderItems->count() ?? 0);
                $sheet->setCellValue('A8', 'Total Grupos:');
                $sheet->setCellValue('B8', count($this->packingList));
                
                // Calcular total de cajas únicas
                $allBoxes = [];
                foreach ($this->packingList as $item) {
                    $allBoxes = array_merge($allBoxes, $item['cajas']);
                }
                $sheet->setCellValue('A9', 'Total Cajas:');
                $sheet->setCellValue('B9', count(array_unique($allBoxes)));
                
                // Estilos para el encabezado
                $sheet->getStyle('A1')->applyFromArray([
                    'font' => ['bold' => true, 'size' => 14],
                ]);
                
                // Estilos para los encabezados de la tabla (fila 11 después de insertar 10 filas)
                $sheet->getStyle('A11:S11')->applyFromArray([
                    'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => '4472C4']
                    ],
                    'alignment' => [
                        'horizontal' => Alignment::HORIZONTAL_CENTER,
                        'vertical' => Alignment::VERTICAL_CENTER,
                    ],
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => Border::BORDER_THIN,
                        ],
                    ],
                ]);
                
                // Aplicar bordes a todas las celdas de datos
                $lastRow = $sheet->getHighestRow();
                if ($lastRow > 11) {
                    $sheet->getStyle('A11:S' . $lastRow)->applyFromArray([
                        'borders' => [
                            'allBorders' => [
                                'borderStyle' => Border::BORDER_THIN,
                            ],
                        ],
                    ]);
                }
                
                // Ajustar altura de filas
                $sheet->getRowDimension(1)->setRowHeight(20);
                $sheet->getRowDimension(11)->setRowHeight(25);
            },
        ];
    }
}

