<?php

namespace App\Console\Commands;

use App\Models\Inventory;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ReviewIncompleteArticles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:review-incomplete
                            {--dry-run : Show what would be marked without actually updating}
                            {--force : Mark articles without confirmation}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Revisa la base de datos y marca artículos con información incompleta como "needs_review"';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Iniciando revisión de artículos con información incompleta...');
        $this->newLine();

        // Criterios para considerar un artículo como incompleto
        $incompleteArticles = Inventory::where(function ($query) {
            $query->whereNull('seccion')
                ->orWhereNull('familia_articulo_description')
                ->orWhereNull('cadena')
                ->orWhereNull('precio_pvp_maximo_temporada')
                ->orWhere('precio_pvp_maximo_temporada', 0)
                ->orWhereNull('peso_unitario')
                ->orWhere('peso_unitario', 0);
        })
            ->where('needs_review', false) // Solo los que no están ya marcados
            ->whereIn('status', ['disponible', 'available']) // Solo artículos disponibles
            ->get();

        if ($incompleteArticles->isEmpty()) {
            $this->info('✅ No se encontraron artículos con información incompleta.');
            return Command::SUCCESS;
        }

        $this->warn("📊 Se encontraron {$incompleteArticles->count()} artículos con información incompleta:");
        $this->newLine();

        // Agrupar por tipo de problema
        $issues = [
            'sin_seccion' => $incompleteArticles->whereNull('seccion')->count(),
            'sin_familia' => $incompleteArticles->whereNull('familia_articulo_description')->count(),
            'sin_cadena' => $incompleteArticles->whereNull('cadena')->count(),
            'sin_precio' => $incompleteArticles->where(function ($item) {
                return is_null($item->precio_pvp_maximo_temporada) || $item->precio_pvp_maximo_temporada == 0;
            })->count(),
            'sin_peso' => $incompleteArticles->where(function ($item) {
                return is_null($item->peso_unitario) || $item->peso_unitario == 0;
            })->count(),
        ];

        $this->table(
            ['Problema', 'Cantidad'],
            [
                ['Sin Sección', $issues['sin_seccion']],
                ['Sin Familia', $issues['sin_familia']],
                ['Sin Cadena', $issues['sin_cadena']],
                ['Sin Precio o Precio = 0', $issues['sin_precio']],
                ['Sin Peso o Peso = 0', $issues['sin_peso']],
            ]
        );

        $this->newLine();

        // Mostrar algunos ejemplos
        $this->info('📋 Ejemplos de artículos afectados:');
        $examples = $incompleteArticles->take(10);

        $tableData = [];
        foreach ($examples as $item) {
            $problems = [];
            if (is_null($item->seccion)) $problems[] = 'Sin Sección';
            if (is_null($item->familia_articulo_description)) $problems[] = 'Sin Familia';
            if (is_null($item->cadena)) $problems[] = 'Sin Cadena';
            if (is_null($item->precio_pvp_maximo_temporada) || $item->precio_pvp_maximo_temporada == 0) $problems[] = 'Sin Precio';
            if (is_null($item->peso_unitario) || $item->peso_unitario == 0) $problems[] = 'Sin Peso';

            $tableData[] = [
                $item->id,
                $item->mocaco,
                $item->full_barcode,
                $item->conteneur ?? 'N/A',
                implode(', ', $problems),
            ];
        }

        $this->table(
            ['ID', 'Mocaco', 'Código de Barras', 'Contenedor', 'Problemas'],
            $tableData
        );

        if ($incompleteArticles->count() > 10) {
            $this->warn("... y " . ($incompleteArticles->count() - 10) . " más.");
        }

        $this->newLine();

        // Modo dry-run
        if ($this->option('dry-run')) {
            $this->info('🔍 Modo DRY-RUN: No se realizarán cambios en la base de datos.');
            $this->info("Se marcarían {$incompleteArticles->count()} artículos como 'needs_review'.");
            return Command::SUCCESS;
        }

        // Confirmación
        if (!$this->option('force')) {
            if (!$this->confirm("¿Desea marcar estos {$incompleteArticles->count()} artículos como 'needs_review'?", true)) {
                $this->warn('❌ Operación cancelada.');
                return Command::FAILURE;
            }
        }

        // Actualizar artículos
        $this->info('⏳ Marcando artículos...');

        $bar = $this->output->createProgressBar($incompleteArticles->count());
        $bar->start();

        $updated = 0;
        foreach ($incompleteArticles as $item) {
            try {
                $item->update([
                    'needs_review' => true,
                    'review_notes' => 'Marcado automáticamente por comando: información incompleta',
                ]);
                $updated++;
            } catch (\Exception $e) {
                $this->error("Error al actualizar artículo ID {$item->id}: {$e->getMessage()}");
            }
            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        $this->info("✅ Se marcaron {$updated} artículos como 'needs_review'.");
        $this->info("📝 Puedes revisarlos en: /article-review");

        return Command::SUCCESS;
    }
}
