# Sistema de Gestión de Inventario - Laravel

## Descripción

Este es un sistema robusto de gestión de inventario desarrollado en Laravel que refactoriza y mejora un prototipo existente basado en PHP y MySQL. El sistema está diseñado para cumplir con los requisitos de seguridad y trazabilidad necesarios para operaciones de exportación.

## Características Principales

### 🔐 Seguridad y Trazabilidad
- **Autenticación de usuarios** con Laravel Sanctum
- **Control de acceso basado en roles (ACL)** usando Spatie Laravel Permission
- **Auditoría completa** de todas las operaciones del sistema
- **Trazabilidad** de cada acción logística realizada

### 📦 Ciclo de Pedido Robusto
- **Estados de pedido**: Borrador → Confirmación → Despachado → Histórico
- **Gestión automática de inventario** al confirmar despachos
- **Eliminación precisa** de artículos del inventario al despachar
- **Historial completo** de pedidos por cliente

### 🏗️ Arquitectura Sostenible
- **Separación de responsabilidades** usando Service Classes
- **Modelos Eloquent** con relaciones bien definidas
- **API RESTful** para integración con frontend
- **Middleware personalizado** para auditoría
- **Migraciones de base de datos** versionadas

## Estructura del Proyecto

```
inventory_new/
├── app/
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── AuthController.php          # Autenticación y roles
│   │   │   ├── ScannerController.php       # Escaneo de códigos de barras
│   │   │   ├── InventoryController.php     # Gestión de inventario
│   │   │   ├── OrderController.php         # Gestión de pedidos
│   │   │   └── RelabelingController.php    # Reetiquetado
│   │   └── Middleware/
│   │       └── AuditMiddleware.php         # Middleware de auditoría
│   ├── Models/
│   │   ├── User.php                        # Usuarios con roles
│   │   ├── Article.php                     # Artículos maestros
│   │   ├── Inventory.php                   # Inventario
│   │   ├── Client.php                      # Clientes
│   │   ├── Order.php                       # Pedidos
│   │   ├── OrderItem.php                   # Items de pedidos
│   │   ├── AuditLog.php                    # Logs de auditoría
│   │   └── RelabelingProduct.php           # Productos reetiquetado
│   └── Services/
│       ├── InventoryService.php            # Lógica de negocio inventario
│       └── OrderService.php                # Lógica de negocio pedidos
├── database/
│   ├── migrations/                         # Migraciones de BD
│   └── seeders/
│       └── InitialDataSeeder.php          # Datos iniciales
├── routes/
│   └── api.php                             # Rutas API
└── public/
    └── index.html                           # Interfaz web demo
```

## Instalación

### Requisitos
- PHP 8.1+
- Composer
- MySQL 8.0+
- Node.js (opcional, para frontend)

### Pasos de Instalación

1. **Clonar el proyecto**
```bash
cd /var/www/inventory/inventory_new
```

2. **Instalar dependencias**
```bash
composer install
```

3. **Configurar entorno**
```bash
cp .env.example .env
php artisan key:generate
```

4. **Configurar base de datos en `.env`**
```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=inventory_laravel
DB_USERNAME=your_username
DB_PASSWORD=your_password
```

5. **Ejecutar migraciones**
```bash
php artisan migrate
```

6. **Crear datos iniciales**
```bash
php artisan db:seed --class=InitialDataSeeder
```

7. **Configurar permisos de almacenamiento**
```bash
chmod -R 775 storage bootstrap/cache
```

## Uso del Sistema

### Acceso al Sistema

El sistema incluye una interfaz web demo disponible en `/public/index.html` que demuestra todas las funcionalidades principales.

**Usuarios de prueba:**
- **Admin**: admin@inventory.com / password
- **Manager**: manager@inventory.com / password  
- **Operator**: operator@inventory.com / password
- **Scanner**: scanner@inventory.com / password

### API Endpoints

#### Autenticación
- `POST /api/login` - Iniciar sesión
- `POST /api/logout` - Cerrar sesión
- `GET /api/me` - Información del usuario actual

#### Escáner
- `POST /api/scanner/search-article` - Buscar artículo por código de barras
- `POST /api/scanner/process-scan` - Procesar escaneo y agregar al inventario
- `POST /api/scanner/get-box-content` - Obtener contenido de caja
- `POST /api/scanner/create-article` - Crear nuevo artículo

#### Inventario
- `POST /api/inventory/search` - Buscar en inventario con filtros
- `PUT /api/inventory/update` - Actualizar artículo de inventario
- `DELETE /api/inventory/delete` - Eliminar artículo del inventario
- `GET /api/inventory/statistics` - Estadísticas del inventario
- `POST /api/inventory/export` - Exportar inventario

#### Pedidos
- `GET /api/orders/clients` - Obtener clientes
- `POST /api/orders/save-draft` - Guardar borrador de pedido
- `GET /api/orders/drafts` - Obtener pedidos borrador
- `POST /api/orders/confirm-dispatch` - Confirmar y despachar pedido
- `GET /api/orders/client-orders` - Historial de pedidos por cliente
- `POST /api/orders/cancel` - Cancelar pedido

#### Reetiquetado
- `POST /api/relabeling/search-product` - Buscar producto por código proveedor
- `POST /api/relabeling/save-product` - Guardar producto de reetiquetado
- `POST /api/relabeling/generate-zpl` - Generar código ZPL para impresión
- `GET /api/relabeling/products` - Obtener todos los productos
- `DELETE /api/relabeling/delete-product` - Eliminar producto

## Roles y Permisos

### Roles Disponibles
- **admin**: Acceso completo al sistema
- **manager**: Gestión de inventario y pedidos
- **operator**: Operaciones básicas de inventario
- **scanner**: Solo escaneo y reetiquetado
- **viewer**: Solo lectura

### Permisos por Funcionalidad
- `inventory.*` - Gestión de inventario
- `orders.*` - Gestión de pedidos
- `clients.*` - Gestión de clientes
- `relabeling.*` - Reetiquetado
- `audit.view` - Ver auditoría
- `users.*` - Gestión de usuarios

## Flujo de Trabajo

### 1. Escaneo de Artículos
1. El operador escanea un código de barras
2. El sistema busca el artículo en la base de datos
3. Si existe, muestra la información y permite agregarlo al inventario
4. Si no existe, permite crear un nuevo artículo
5. Se registra la acción en el log de auditoría

### 2. Gestión de Pedidos
1. **Crear Borrador**: Seleccionar cliente y escanear cajas
2. **Confirmar Despacho**: El sistema elimina automáticamente los artículos del inventario
3. **Historial**: Consultar pedidos despachados por cliente

### 3. Reetiquetado
1. Escanear código de proveedor
2. Completar información del producto
3. Generar código ZPL para impresión en impresora Zebra

## Auditoría y Trazabilidad

Todas las operaciones del sistema se registran en la tabla `audit_logs` con:
- Usuario que realizó la acción
- Tipo de acción (create, update, delete, scan, dispatch, etc.)
- Modelo afectado
- Valores anteriores y nuevos
- Timestamp
- IP y User Agent
- Notas adicionales

## Mejoras Implementadas

### Comparado con el Sistema PHP Original

1. **Seguridad**
   - ✅ Autenticación robusta con tokens
   - ✅ Control de acceso basado en roles
   - ✅ Validación de entrada en todas las APIs
   - ✅ Protección CSRF y sanitización

2. **Arquitectura**
   - ✅ Separación de responsabilidades (Controllers, Services, Models)
   - ✅ Uso de Eloquent ORM
   - ✅ Migraciones versionadas
   - ✅ Middleware personalizado

3. **Trazabilidad**
   - ✅ Log completo de todas las operaciones
   - ✅ Auditoría de cambios en datos
   - ✅ Trazabilidad de usuarios y acciones

4. **Gestión de Pedidos**
   - ✅ Estados claros (Draft → Confirmed → Dispatched)
   - ✅ Eliminación automática del inventario al despachar
   - ✅ Historial completo de pedidos
   - ✅ Validaciones de integridad

5. **Escalabilidad**
   - ✅ API RESTful para integración
   - ✅ Servicios reutilizables
   - ✅ Estructura modular
   - ✅ Preparado para microservicios

## Desarrollo y Mantenimiento

### Comandos Útiles

```bash
# Crear nueva migración
php artisan make:migration create_table_name

# Crear nuevo modelo
php artisan make:model ModelName

# Crear nuevo controlador
php artisan make:controller ControllerName

# Crear nuevo servicio
mkdir -p app/Services && touch app/Services/ServiceName.php

# Ejecutar tests
php artisan test

# Limpiar caché
php artisan cache:clear
php artisan config:clear
php artisan route:clear
```

### Estructura de Base de Datos

El sistema utiliza las siguientes tablas principales:
- `users` - Usuarios del sistema
- `roles` / `permissions` - Sistema de roles (Spatie)
- `articles` - Catálogo de artículos
- `inventory` - Inventario actual
- `clients` - Clientes
- `orders` - Pedidos
- `order_items` - Items de pedidos
- `audit_logs` - Log de auditoría
- `relabeling_products` - Productos para reetiquetado

## Contribución

Para contribuir al proyecto:

1. Fork el repositorio
2. Crear una rama para la nueva funcionalidad
3. Implementar cambios siguiendo las convenciones de Laravel
4. Agregar tests si es necesario
5. Crear un Pull Request

## Licencia

Este proyecto está bajo la licencia MIT. Ver el archivo LICENSE para más detalles.

## Soporte

Para soporte técnico o preguntas sobre el sistema, contactar al equipo de desarrollo.